import qs, { ParsedUrlQueryInput } from 'querystring';

import axios, { AxiosResponse } from 'axios';

import { Injectable } from '@nestjs/common';
import { HttpService, RequestConfig } from '@server/shared/http';
import { Context } from '@yandex-int/nest-common';

import {
  BaseTrackState,
  BindPrimaryPhoneTrackState,
  ConfirmPhoneTrackState,
  SendCodeToBindPrimaryPhoneData,
  SendCodeToConfirmPhoneData,
  VerifyCodeToBindPrimaryPhoneData,
  VerifyCodeToConfirmPhoneData,
} from './api-types';
import {
  PassportApiResponse,
  SendCodeToBindPrimaryPhoneResponse,
  SendCodeToConfirmPhoneResponse,
  VerifyCodeToBindPrimaryPhoneResponse,
  VerifyCodeToConfirmPhoneResponse,
} from './passport.interface';

interface GetRequestOptions<T = void> {
  url: string;
  query?: ParsedUrlQueryInput;
  config?: RequestConfig<T>;
}

interface PostRequestOptions<T> extends GetRequestOptions<T> {
  data?: T;
}

@Injectable()
export class PassportPhoneService {
  constructor(private context: Context, private http: HttpService) {}

  private getHeaders(overriddenHeaders?: Record<string, string | number | boolean | undefined>) {
    const headers: Record<string, string | number | boolean | undefined> = {
      'Ya-Consumer-Client-Ip': this.context.req.ip,
      'Ya-Client-Host': this.context.req.headers.host,
      'Ya-Client-User-Agent': this.context.req.headers['user-agent'],
      ...overriddenHeaders,
    };

    return headers;
  }

  private get<T = any>(options: GetRequestOptions) {
    const { url, query, config = {} } = options;

    return this.http.get<T, AxiosResponse<T, void>>(url, {
      ...config,
      params: { consumer: 'passport', ...config.params, ...query },
      headers: this.getHeaders(config.headers),
    });
  }

  private post<T = any, D = any>(options: PostRequestOptions<D>) {
    const { url, query, data, config = {} } = options;

    return this.http.post<T, AxiosResponse<T, D>>(
      url,
      data ? qs.stringify(data as unknown as ParsedUrlQueryInput) : undefined,
      {
        ...config,
        params: {
          consumer: 'passport',
          ...config.params,
          ...query,
        },
        headers: this.getHeaders(config.headers),
      },
    );
  }

  async getTrackState(trackId: string) {
    try {
      const response = await this.get<
        PassportApiResponse<BaseTrackState | BindPrimaryPhoneTrackState | ConfirmPhoneTrackState>
      >({
        url: `/1/track/${trackId}/`,
      });

      return response.data;
    } catch (err) {
      // Если передать невалидный трек, то Паспорт вернет 400 статус, а axios выкинет ошибку
      // Поэтому обработаем случай, когда в response есть data, иначе прокинем ошибку выше
      if (axios.isAxiosError(err) && err.response?.data) {
        return err.response.data;
      }

      throw err;
    }
  }

  async sendCodeToConfirmPhone(params: SendCodeToConfirmPhoneData) {
    const lang = this.context.req.langdetect.id;
    const { track_id: trackId, display_language: displayLanguage = lang, ...rest } = params;
    const data: SendCodeToConfirmPhoneData = {
      display_language: displayLanguage,
      ...rest,
    };

    // NOTE: Если передать null или undefined, то бэк будет возвращать ошибку
    if (trackId) {
      data.track_id = trackId;
    }

    const response = await this.post<SendCodeToConfirmPhoneResponse, SendCodeToConfirmPhoneData>({
      url: '/1/bundle/phone/confirm/submit/',
      data,
    });

    return response.data;
  }

  async verifyCodeToConfirmPhone(data: VerifyCodeToConfirmPhoneData) {
    const response = await this.post<
      VerifyCodeToConfirmPhoneResponse,
      VerifyCodeToConfirmPhoneData
    >({
      url: '/1/bundle/phone/confirm/commit/',
      data,
    });

    return response.data;
  }

  async sendCodeToBindPrimaryPhone(params: SendCodeToBindPrimaryPhoneData) {
    const lang = this.context.req.langdetect.id;
    const { track_id: trackId, display_language: displayLanguage = lang, ...rest } = params;
    const data: SendCodeToBindPrimaryPhoneData = {
      display_language: displayLanguage,
      ...rest,
    };

    // NOTE: Если передать null или undefined, то бэк будет возвращать ошибку
    if (trackId) {
      data.track_id = trackId;
    }

    const response = await this.post<
      SendCodeToBindPrimaryPhoneResponse,
      SendCodeToBindPrimaryPhoneData
    >({
      url: '/2/bundle/phone/confirm_and_bind_secure/submit/',
      data,
    });

    return response.data;
  }

  async verifyCodeToBindPrimaryPhone(data: VerifyCodeToBindPrimaryPhoneData) {
    const response = await this.post<
      VerifyCodeToBindPrimaryPhoneResponse,
      VerifyCodeToBindPrimaryPhoneData
    >({
      url: '/2/bundle/phone/confirm_and_bind_secure/commit/',
      data,
    });

    return response.data;
  }

  async toggleLoginWithPasswordAndSms(value: boolean) {
    const response = await this.post<PassportApiResponse>({
      url: `/2/account/options/`,
      data: { sms_2fa_on: value },
    });

    return response.data;
  }
}
