import qs, { ParsedUrlQueryInput } from 'querystring';

import { AxiosResponse } from 'axios';

import { Injectable } from '@nestjs/common';
import { HttpService, RequestConfig } from '@server/shared/http';
import { Context } from '@yandex-int/nest-common';

import { IPassportPublicIdValidate, IPublicIdValidateParams } from './api-types/public-id-validate';
import { ValidatePublicIdResponse } from './passport.interface';

interface GetRequestOptions<T = void> {
  url: string;
  query?: ParsedUrlQueryInput;
  config?: RequestConfig<T>;
}

interface PostRequestOptions<T> extends GetRequestOptions<T> {
  data?: T;
}

@Injectable()
export class PassportPublicIdService {
  constructor(private context: Context, private http: HttpService) {}

  private getHeaders(overriddenHeaders?: Record<string, string | number | boolean | undefined>) {
    const headers: Record<string, string | number | boolean | undefined> = {
      'Ya-Consumer-Client-Ip': this.context.req.ip,
      'Ya-Client-Host': this.context.req.headers.host,
      'Ya-Client-User-Agent': this.context.req.headers['user-agent'],
      ...overriddenHeaders,
    };

    return headers;
  }

  private post<T = any, D = any>(options: PostRequestOptions<D>) {
    const { url, query, data, config = {} } = options;

    return this.http.post<T, AxiosResponse<T, D>>(
      url,
      data ? qs.stringify(data as unknown as ParsedUrlQueryInput) : undefined,
      {
        ...config,
        params: {
          consumer: 'passport',
          ...config.params,
          ...query,
        },
        headers: this.getHeaders(config.headers),
      },
    );
  }

  async validatePublicId(publicIdData: IPublicIdValidateParams) {
    const { publicId, trackId } = publicIdData;

    const data: IPassportPublicIdValidate = {
      public_id: publicId,
      track_id: trackId,
      force_clean_web: true,
    };

    const response = await this.post<ValidatePublicIdResponse, IPassportPublicIdValidate>({
      url: '/1/bundle/validate/public_id/',
      data,
    });

    return response.data;
  }
}
