import qs, { ParsedUrlQueryInput } from 'querystring';

import { AxiosResponse } from 'axios';
import type { Express } from 'express';
import FormData from 'form-data';

import { Injectable, InternalServerErrorException } from '@nestjs/common';
import { BlackboxService } from '@server/shared/blackbox';
import { HttpService, RequestConfig } from '@server/shared/http';
import { Context, StorageMemoize } from '@yandex-int/nest-common';

import { AccountUpdateParams, PassportAccountUpdate, SessionData, TrackData } from './api-types';
import { ISuggestLoginParams } from './api-types/login-suggest';
import { IPassportProfileUpdate, IProfileUpdateParams } from './api-types/profile-update';
import { ValidationFieldName } from './api-types/validation-field-name';
import {
  AccountAuthEventsResponse,
  AccountEventsResponse,
  BillingData,
  BillingListResponse,
  CreateBindingProps,
  CreateBindingResponse,
  DoBindingResponse,
  FamilyDataResponse,
  LegacyPassportData,
  PassportApiResponse,
  ProfileUpdateResponse,
  SocialProfilesInitResponse,
  SuggestLogin,
} from './passport.interface';

const isVISA = (system = '') => system.toLowerCase().includes('visa');

@Injectable()
export class PassportService {
  constructor(
    public http: HttpService,
    private context: Context,
    private blackbox: BlackboxService,
  ) {}

  private get<T = any, U = AxiosResponse<T>, D = any>(
    pathname: string,
    config: RequestConfig<D> = {},
  ): Promise<U> {
    return this.http.get<T, U>(`${pathname}?consumer=passport`, {
      ...config,
      headers: {
        'Ya-Client-Cookie': this.context.req.headers.cookie,
        'Ya-Client-Host': this.context.req.headers.host,
        ...config.headers,
      },
    });
  }

  @StorageMemoize()
  async getLegacyPassportData(): Promise<LegacyPassportData> {
    const response = await this.get<LegacyPassportData>('/1/bundle/account/');

    if (response.data.status === 'error') {
      throw new InternalServerErrorException(response.data.errors);
    }

    return response.data;
  }

  async getAccountAuthEvents(period = 180) {
    const response = await this.get<AccountAuthEventsResponse>('/2/bundle/account/history/', {
      params: {
        hours_limit: period * 24,
        password_auths: true,
      },
    });

    return response.data;
  }

  async getAccountEvents(period = 180) {
    const response = await this.get<AccountEventsResponse>('/1/bundle/account/events/', {
      params: {
        hours_limit: period * 24,
      },
    });

    return response.data;
  }

  async getBillingListPaymentMethods() {
    const trackId = await this.getTrackId();

    const response = await this.http.post<BillingListResponse>(
      '/1/bundle/billing/list_payment_methods/?consumer=passport',
      qs.stringify({ track_id: trackId }),
      {
        headers: {
          'Ya-Consumer-Client-Ip': this.context.req.ip,
          'Ya-Client-Cookie': this.context.req.headers.cookie,
          'Ya-Client-Host': this.context.req.headers.host,
          'Ya-Client-User-Agent': this.context.req.headers['user-agent'],
        },
      },
    );

    if (response.data.status === 'error') {
      return { status: 'error', cards: [], trackId };
    }

    const { payment_methods: paymentMethods, status } = response.data.billing_response;

    const data: BillingData = { status, cards: [], trackId };

    Object.keys(paymentMethods).forEach((item) => {
      const method = paymentMethods[item];

      const isFamilyCard = Boolean(method.payer_info && method.payer_info.uid);
      const familyAdminUid = Number(method.payer_info && method.payer_info.uid);

      if (method.type === 'card') {
        data.cards.push({
          // потому что может быть visa electron ...
          system: isVISA(method.system) ? 'VISA' : method.system,
          number: method.number,
          name: method.name,
          id: method.id,
          cardId: method.card_id,
          proto: method.proto,
          isFamilyCard,
          familyAdminUid,
        });
      }
    });

    return data;
  }

  async createBinding({ trackId, returnPath, lang, domainSfx, templateTag }: CreateBindingProps) {
    const response = await this.http.post<CreateBindingResponse>(
      '/1/bundle/billing/create_binding/?consumer=passport',
      qs.stringify({
        track_id: trackId,
        return_path: returnPath,
        domain_sfx: domainSfx,
        template_tag: templateTag,
        lang,
      }),
      {
        headers: {
          'Ya-Consumer-Client-Ip': this.context.req.ip,
          'Ya-Client-Cookie': this.context.req.headers.cookie,
          'Ya-Client-Host': this.context.req.headers.host,
          'Ya-Client-User-Agent': this.context.req.headers['user-agent'],
        },
      },
    );

    if (response.data.status === 'error') {
      return { status: 'error' };
    }

    const token = response.data.billing_response.purchase_token;

    return this.doBinding(trackId, token);
  }

  async doBinding(trackId: string, token: string) {
    const response = await this.http.post<DoBindingResponse>(
      '/1/bundle/billing/do_binding/?consumer=passport',
      qs.stringify({
        track_id: trackId,
        purchase_token: token,
      }),
      {
        headers: {
          'Ya-Consumer-Client-Ip': this.context.req.ip,
          'Ya-Client-Cookie': this.context.req.headers.cookie,
          'Ya-Client-Host': this.context.req.headers.host,
          'Ya-Client-User-Agent': this.context.req.headers['user-agent'],
        },
      },
    );

    if (response.data.status === 'error') {
      return { status: 'error' };
    }

    const { _TARGET, purchase_token } = response.data.billing_response.binding_form;

    const url = new URL(_TARGET);

    url.searchParams.append('purchase_token', purchase_token);

    return { status: 'success', url: url.toString() };
  }

  async unbindCard(card: string) {
    const trackId = await this.getTrackId();

    const response = await this.http.post<BillingListResponse>(
      '/1/bundle/billing/unbind_card/?consumer=passport',
      qs.stringify({ track_id: trackId, card }),
      {
        headers: {
          'Ya-Consumer-Client-Ip': this.context.req.ip,
          'Ya-Client-Cookie': this.context.req.headers.cookie,
          'Ya-Client-Host': this.context.req.headers.host,
          'Ya-Client-User-Agent': this.context.req.headers['user-agent'],
        },
      },
    );

    if (response.data.status === 'error') {
      return { status: 'error' };
    }

    return { status: 'success' };
  }

  @StorageMemoize()
  async getSocialProfiles() {
    const response = await this.get<SocialProfilesInitResponse>('/1/change_social/init/', {
      headers: {
        'Ya-Consumer-Client-Ip': this.context.req.ip,
      },
    });

    return response.data;
  }

  @StorageMemoize()
  async suggestLogin(data: ISuggestLoginParams) {
    const response = await this.get<SuggestLogin>('/1/bundle/suggest/login/', {
      headers: {
        'Ya-Consumer-Client-Ip': this.context.req.ip,
        'Ya-Client-User-Agent': this.context.req.get('User-Agent'),
      },
      params: {
        track_id: data.trackId,
        first_name: data.firstName,
        last_name: data.lastName,
        login: data.login,
        language: data.language,
      },
    });

    return response.data;
  }

  /**
   *
   * @see https://wiki.yandex-team.ru/passport/api/bundle/changeaccount#izmenitpersonalnyedannyebandlovajaversija
   * @param profileData
   * @returns
   */
  async updateProfile(profileData: IProfileUpdateParams) {
    if (!profileData.trackId) {
      profileData.trackId = await this.getTrackId();
    }

    const {
      publicId,
      trackId,
      firstname,
      lastname,
      gender,
      country,
      city,
      timezone,
      displayName,
      birthday,
    } = profileData;

    const data: IPassportProfileUpdate = {
      public_id: publicId,
      track_id: trackId,
      firstname,
      lastname,
      gender,
      country,
      city,
      timezone,
      birthday,
      display_name: displayName,
    };

    (Object.keys(data) as (keyof IPassportProfileUpdate)[]).forEach((key) => {
      if (data[key] === undefined) {
        delete data[key];
      }
    });

    const response = await this.http.post<ProfileUpdateResponse>(
      `/1/bundle/account/person/?consumer=passport`,
      qs.stringify(data as ParsedUrlQueryInput),
      {
        headers: {
          'Ya-Client-Cookie': this.context.req.headers.cookie,
          'Ya-Client-Host': this.context.req.headers.host,
          'Ya-Consumer-Client-Ip': this.context.req.ip,
        },
      },
    );

    return response.data;
  }

  async getAccountWithFamilyData() {
    const response = await this.get<FamilyDataResponse>('/1/bundle/account/', {
      params: {
        need_display_name_variants: false,
        need_phones: true,
        need_emails: false,
        need_social_profiles: false,
        need_question: false,
        need_additional_account_data: false,
        need_family_info: true,
        need_family_members: true,
        need_family_kids: true,
        need_family_invites: true,
      },
    });

    return response;
  }

  async updateAccount(accountData: AccountUpdateParams) {
    const data: PassportAccountUpdate = {};

    if (accountData.authWithQr !== undefined) {
      data.qr_code_login_forbidden = !accountData.authWithQr;
    }

    if (Object.keys(data).length === 0) {
      return false;
    }

    const response = await this.http.post<PassportApiResponse>(
      `/2/account/options/?consumer=passport`,
      qs.stringify(data as ParsedUrlQueryInput),
    );

    return response.data.status === 'ok';
  }

  async getTrackId(): Promise<string> {
    const blackbox = await this.blackbox.getBlackbox();
    const response = await this.http.post<PassportApiResponse<TrackData>>(
      '/1/bundle/track/init/?consumer=passport',
      qs.stringify({ uid: blackbox.uid }),
      {
        headers: {
          'Ya-Client-Cookie': this.context.req.headers.cookie,
          'Ya-Client-Host': this.context.req.headers.host,
          'Ya-Consumer-Client-Ip': this.context.req.ip,
        },
      },
    );

    if (response.data.status === 'error') {
      throw new InternalServerErrorException(response.data.errors);
    }

    return response.data.track_id;
  }

  async getAvatarTrackId(): Promise<string> {
    const response = await this.http.get<PassportApiResponse<TrackData>>(
      '/1/change_avatar/init/?consumer=passport',
      {
        headers: {
          'Ya-Client-Cookie': this.context.req.headers.cookie,
          'Ya-Client-Host': this.context.req.headers.host,
          'Ya-Consumer-Client-Ip': this.context.req.ip,
        },
      },
    );

    if (response.data.status === 'error') {
      throw new InternalServerErrorException(response.data.errors);
    }

    return response.data.track_id;
  }

  async updateAuthSession(trackId: string) {
    const response = await this.http.post<PassportApiResponse<SessionData>>(
      '/1/bundle/session/?consumer=passport',
      qs.stringify({ track_id: trackId }),
      {
        headers: {
          'Ya-Client-Cookie': this.context.req.headers.cookie,
          'Ya-Client-Host': this.context.req.headers.host,
          'Ya-Consumer-Client-Ip': this.context.req.ip,
        },
      },
    );

    if (response.data.status === 'error') {
      throw new InternalServerErrorException(response.data.errors);
    }

    this.context.res.setHeader('Set-Cookie', response.data.cookies);

    return response.data;
  }

  async addAvatar(avatar: Express.Multer.File, isDefault?: boolean) {
    const trackId = await this.getAvatarTrackId();
    const formData = new FormData();

    if (isDefault) {
      formData.append('default', 'true');
    }

    formData.append('track_id', trackId);
    formData.append('file', avatar.buffer, avatar.originalname);
    const response = await this.http.post<PassportApiResponse<void>>(
      `/1/change_avatar/?consumer=passport`,
      formData,
      {
        headers: {
          'Ya-Client-Cookie': this.context.req.headers.cookie,
          'Ya-Client-Host': this.context.req.headers.host,
          'Ya-Consumer-Client-Ip': this.context.req.ip,
          ...formData.getHeaders(),
        },
      },
    );

    return response.data;
  }

  async deleteDefaultAvatar() {
    const trackId = await this.getAvatarTrackId();

    const response = await this.http.delete<PassportApiResponse<void>>(
      `/1/change_avatar/default/?consumer=passport&track_id=${trackId}`,
      {
        headers: {
          'Ya-Client-Cookie': this.context.req.headers.cookie,
          'Ya-Client-Host': this.context.req.headers.host,
          'Ya-Consumer-Client-Ip': this.context.req.ip,
        },
      },
    );

    return response.data;
  }

  async validateProfileField(fieldName: ValidationFieldName, fieldValue: string) {
    const trackId = await this.getTrackId();
    const data = {
      track_id: trackId,
      [fieldName]: fieldValue,
    };

    const response = await this.http.post<PassportApiResponse<void>>(
      `/1/bundle/validate/${fieldName}/?consumer=passport`,
      qs.stringify(data),
    );

    return response.data;
  }
}
