import { AsyncResource } from 'async_hooks';
import EventEmitter from 'events';

import { NextFunction, Request, Response } from 'express';

import { Injectable, NestMiddleware } from '@nestjs/common';
import { staticLogger } from '@server/shared/logger/logger.service';

function safeGetPath(url: string) {
  try {
    return new URL(url, 'https://passport.yandex.ru').pathname;
  } catch (err) {
    staticLogger.error('Cannot parse url', { err });

    return '';
  }
}

@Injectable()
export class RequestEventsMiddleware extends EventEmitter implements NestMiddleware {
  protected run(eventName: string, arg: Object) {
    // eslint-disable-next-line array-callback-return
    this.listeners(eventName).forEach((handler) => handler(arg));
  }

  private asyncResourceBind<T extends (...args: any[]) => any>(fn: T) {
    // NOTE: Взято из исходников, так как в node@12.18.1 нет статического метода `AsyncResource.bind`
    // https://github.com/nodejs/node/blob/v14.19.3/lib/async_hooks.js#L222-L246
    const asyncResource = new AsyncResource(fn.name || 'bound-anonymous-fn');
    const ret = asyncResource.runInAsyncScope.bind(asyncResource, fn);

    Object.defineProperties(ret, {
      length: {
        configurable: true,
        enumerable: false,
        value: fn.length,
        writable: false,
      },
      asyncResource: {
        configurable: true,
        enumerable: true,
        value: this,
        writable: true,
      },
    });

    return ret as T & { asyncResource: AsyncResource };
  }

  async use(req: Request, res: Response, next: NextFunction) {
    let path = safeGetPath(req.originalUrl);

    if (!path || path.indexOf('/_next/static') === 0 || path.indexOf('/_next/image') === 0) {
      return next();
    }

    // чтобы хоть как то различать ручки graphql, записываем с какой страницы мы пришли в эту ручку
    if (path.indexOf('/graphql') >= 0) {
      const referrer = safeGetPath(req.header('referer') || '');

      path += referrer;
    }

    const start = process.hrtime.bigint();
    let finished = false;

    this.run('start', { start, finished, path, req, res });

    res.once(
      'finish',
      this.asyncResourceBind(() => {
        finished = true;

        this.run('finish', { start, finished, path, req, res });
      }),
    );

    res.once(
      'error',
      this.asyncResourceBind((err: Error) => {
        if (!finished) {
          finished = true;
          this.run('error', { start, finished, path, err, req, res });
        }
      }),
    );

    res.once(
      'close',
      this.asyncResourceBind(() => {
        if (!finished) {
          finished = true;
          this.run('close', { start, finished, path, req, res });
        }
      }),
    );

    next();
  }
}
