import { DynamicModule, Module, Provider } from '@nestjs/common';

import {
  StaticMapModuleAsyncOptions,
  StaticMapModuleOptions,
  StaticMapOptionsFactory,
} from './interfaces';
import { DEFAULT_STATIC_MAP_OPTIONS, STATIC_MAP_OPTIONS_TOKEN } from './static-map.constants';
import { StaticMapService } from './static-map.service';

@Module({
  providers: [
    StaticMapService,
    { provide: STATIC_MAP_OPTIONS_TOKEN, useValue: DEFAULT_STATIC_MAP_OPTIONS },
  ],
  exports: [StaticMapService],
})
export class StaticMapModule {
  static forRoot(options: StaticMapModuleOptions) {
    const { isGlobal, ...StaticMapOptions } = options;

    return {
      module: StaticMapModule,
      global: isGlobal,
      providers: [{ provide: STATIC_MAP_OPTIONS_TOKEN, useValue: StaticMapOptions }],
    };
  }

  static forRootAsync(options: StaticMapModuleAsyncOptions): DynamicModule {
    return {
      module: StaticMapModule,
      global: options.isGlobal,
      imports: options.imports,
      providers: this.createAsyncProviders(options),
    };
  }

  private static createAsyncProviders(options: StaticMapModuleAsyncOptions): Provider[] {
    const providers: Provider[] = [this.createAsyncOptionsProvider(options)];

    if (!options.useExisting && options.useClass) {
      providers.push({ provide: options.useClass, useClass: options.useClass });
    }

    return providers;
  }

  private static createAsyncOptionsProvider(options: StaticMapModuleAsyncOptions): Provider {
    const { useFactory, useExisting, useClass, inject = [] } = options;

    if (useFactory) {
      return {
        provide: STATIC_MAP_OPTIONS_TOKEN,
        useFactory,
        inject,
      };
    }

    const dep = useExisting || useClass;

    if (dep) {
      return {
        provide: STATIC_MAP_OPTIONS_TOKEN,
        async useFactory(optionsFactory: StaticMapOptionsFactory) {
          return await optionsFactory.createStaticMapOptions();
        },
        inject: [dep],
      };
    }

    throw new Error('Invalid configuration. Must provide useFactory, useClass or useExisting');
  }
}
