import crypto from 'crypto';

import { Inject, Injectable } from '@nestjs/common';

import type { StaticMapOptions, StaticMapUrlOptions } from './interfaces';
import { STATIC_MAP_OPTIONS_TOKEN } from './static-map.constants';

@Injectable()
export class StaticMapService {
  private options: StaticMapOptions;

  constructor(@Inject(STATIC_MAP_OPTIONS_TOKEN) options: StaticMapOptions) {
    this.options = options;
  }

  getUrl(options: StaticMapUrlOptions) {
    const { host, apiKey, secretKey } = this.options;

    const prefix = '/1.x/?';
    const queryParams = this.getQueryParams(options);

    if (apiKey && secretKey) {
      const signature = this.createSignature(`${prefix}${queryParams.toString()}`, secretKey);

      queryParams.set('signature', signature);
    }

    return `//${host}${prefix}${queryParams}`;
  }

  private getQueryParams(options: StaticMapUrlOptions) {
    const {
      layout = ['map'],
      center,
      zoom,
      scale,
      span,
      size,
      lang,
      hideLogo = true,
      hideCopyright = true,
      hideText = false,
      colorSchema,
    } = options;
    const queryParams = new URLSearchParams();

    queryParams.set('l', layout.join(','));

    if (center) {
      queryParams.set('ll', `${center.lng},${center.lat}`);
    }

    if (typeof zoom === 'number') {
      queryParams.set('z', Math.min(Math.max(zoom, 0), 23).toString());
    }

    if (span) {
      queryParams.set('spn', `${span.lngDiff},${span.latDiff}`);
    }

    if (scale) {
      queryParams.set('scale', Math.min(Math.max(scale, 1), 4).toString());
    }

    if (size) {
      queryParams.set('size', `${size.width},${size.height}`);
    }

    if (lang) {
      queryParams.set('lang', lang);
    }

    if (this.options.apiKey && this.options.secretKey) {
      queryParams.set('api_key', this.options.apiKey);

      if (hideLogo) {
        queryParams.set('lg', '0');
      }

      if (hideCopyright) {
        queryParams.set('cr', '0');
      }

      if (hideText) {
        queryParams.set('style', 'elements:label%7Cstylers.visibility:off');
      }

      if (colorSchema) {
        queryParams.set('theme', colorSchema);
      }
    }

    return queryParams;
  }

  private createSignature(data: string, secret: string) {
    const base64secret = Buffer.from(secret, 'base64');
    const signature = crypto.createHmac('sha256', base64secret).update(data).digest('base64');

    return signature;
  }
}
