declare module 'luster' {
  import EventEmitter = require('events');

  export interface BunkerData<TBunkerData> {
    getNode<TNode>(path?: string): TNode;
    getNode(p: undefined): TBunkerData;
  }

  export interface LusterConfigControl {
    /** Сколько времени от форка ждать появления события online прежде чем считать запуск неудачным (мс) */
    forkTimeout: number;

    /**
     * Сколько времени ждать от момента disconnect до сообщения exit от воркера,
     * прежде чем его завершить принужительно (мс)
     * */
    stopTimeout: number;

    /** Сколько времени воркер должен проработать, чтобы не считать циклически упавшим (мс) */
    exitThreshold: number;

    /** Сколько раз пытаться запускать воркеры, прежде чем считать запуск невозможным */
    allowedSequentialDeaths: number;
  }

  export interface LusterConfig {
    /** Путь к файлу с кодом воркеров */
    app: string;

    /** Количество воркеров. Если не задано, то запускается по количеству ядер процессоров */
    workers?: number;

    control: LusterConfigControl;

    /** Задаёт группы веб-воркеров */
    server?: {
      /**
       * Начальный порт группы воркеров
       * У каждой следующей группы будет порт + 1
       * значение порта пробрасывается в соответствующие процессы воркеров через переменную среды port
       * server.listen(process.env.port)
       */
      port: number;

      /**
       * Количество групп воркеров
       * У каждой группы будет свой порт (port + group number)
       */
      groups: number;
    };

    /**
     * Конфигурация плагинов
     * Каждый ключ — название модуля в node_modules или в папке заданной extensionsPath
     */
    extensions?: Record<string, unknown>;

    /**
     * Абсолютный путь к каталогу с плагинами
     * (если плагины не могут быть найдены по относительгному пути от файла воркера)
     */
    extensionsPath?: string;

    /** Предельное время ожидания загрузки всех плагинов */
    extensionsLoadTimeout?: number;
  }

  type RpcHandler<TMessage = unknown> = (sender: LusterProcess, message: TMessage) => void;

  type CallbackHandler<TData, TError = unknown> = (
    sender: LusterProcess,
    err: TError,
    message: TData,
  ) => void;

  export interface WorkerProcess extends EventEmitter {
    isMaster: false;
    isWorker: true;
    wid: string;
    ready: boolean;
    remoteCall<TData>(string, ...data: TData);
    remoteCallWithCallback<TData, TResponse>(opts: {
      command: string;
      callback: CallbackHandler<TResponse>;
      timeout?: number;
      data?: TData;
    }): void;
    registerRemoteCommand<TMessage = unknown>(name: string, fn: RpcHandler<TMessage>): void;
    restart(): void;
    configure(config: LusterConfig, applyEnv?: boolean, basedir?: string): this;
  }

  export interface MasterProcess extends WorkerProcess {
    isMaster: true;
    isWorker: false;
    getWorkersArray(): WorkerProcess[];
    softRestart(): void;
    shutdown(): this;
    run(): void;
  }

  export type ExtensionConfig<TConfig> = {
    get: <TKey extends keyof TConfig>(key: TKey) => TConfig[key];
  };

  export type LusterProcess = WorkerProcess | MasterProcess;

  export const bunker: {
    configureAPI<TBunkerData>(): BunkerData<TBunkerData>;
  };

  const worker: MasterProcess;
  export default worker;
}
