#!/bin/bash

set -e;
set -x;

SCRIPT_PATH=$(cd "$(dirname ${BASH_SOURCE[0]})"; pwd -P);
SERVICE_ROOT=$(cd "${SCRIPT_PATH}/../"; pwd -P);

# Указываем TTY для ввода Passphrase
export GPG_TTY=$(tty);

function confirm {
    echo "$1"
    read -r -n 1 -p "Confirm - [y/N]" confirm
    echo ""
    echo "$confirm"
    [[ ! $confirm =~ ^[Yy]$ ]] && exit 1
    return 0
}

function initPgpkey {
    GPGDIR=${GPGDIR:-${HOME}/.gnupg}

    if ! pgrep gpg-agent; then
        [[ ! -d ${GPGDIR} ]] && echo ".gnupg not found" && exit 1;

        # форсим загрузку pgp ключей сразу, чтобы на последующих этапах всё проходило гладко
        echo "" | gpg2 -s --homedir ${GPGDIR} --pinentry-mode=loopback;
    else
        gpg2 --list-secret-keys || echo $?;
    fi
}

function initSshkey {
    if [[ ! -d ~/.ssh/ ]]; then
        [[ ! -d $1 ]]  && echo ".ssh not found" && exit 1;

        mkdir -p ~/.ssh;
        cp -rv $1/id_* ~/.ssh/;
        chmod 700 ~/.ssh;
        chmod 600 ~/.ssh/*;
        grep "StrictHostKeyChecking" ~/.ssh/config || echo -e "Host *\n\tStrictHostKeyChecking no" > ~/.ssh/config;
        pkill -fc ssh-agent;
    fi

    if ! pgrep ssh-agent; then
        eval `ssh-agent`;
        ssh-add;
    fi
}

function initUserName {
    local  __RESULT=$1
    [[ -z "$__RESULT" ]] && exit 1

    local USER_NAME=${ROBOT_CI_LOGIN:-$USER}
    [[ -z $USER_NAME ]] && read -r -p "Enter username for git/gpg: " USER_NAME
    [[ -z $USER_NAME ]] && exit 1

    eval $__RESULT="'$USER_NAME'"
}

function initUserEmail {
    local  __RESULT=$1
    [[ -z "$__RESULT" ]] && exit 1

    local USER_EMAIL=${ROBOT_CI_EMAIL:-${USER}@yandex-team.ru}
    [[ -z $USER_EMAIL ]] && read -r -p "Enter email for git/gpg: " USER_EMAIL
    [[ -z $USER_EMAIL ]] && exit 1

    eval $__RESULT="'$USER_EMAIL'"
}

function downloadIpregDB {
    curl -f -o debian/ipreg-base.json -H "Authorization: OAuth ${SANDBOX_TOKEN}" https://proxy.sandbox.yandex-team.ru/last/IPREG_LAYOUT_STABLE
}

function uploadPackages {
    DUPUSER=$(echo $DEBEMAIL | cut -f1 -d@) # extract string before @
    cat ${SERVICE_ROOT}/debian/dupload.tpl | sed "s/LOGIN/$DUPUSER/g" >  ${SERVICE_ROOT}/debian/dupload.conf

    [ -z "$CI" ] && confirm "Push dev-build"

    dupload --configfile ${SERVICE_ROOT}/debian/dupload.conf \
        --force --nomail --to passport-xenial ${SERVICE_ROOT}/../*.changes || echo "Some error :("
}

function genChangelog {
    echo 'Generating changelog...';

    # пока вырубаем всё это, потому что как работать с тегами и версиями в арке не понятно
    PREV_VER='1.19.0'; # $(git tag -l --sort=-v:refname | grep -E 'v[0-9]+\.[0-9]+\.[0-9]+' | head -n1);
    export NEXT_VER=$(npx semver -p --preid $(date +%s) -i preminor ${PREV_VER/[z-a]//})
    # тут выходим в корень монорепы
    # mkdir -p ../../debian;
    # echo "yandex-id-frontend (${PREV_VER/v/}) unstable; urgency=low" > ./changelog;

    cat >./debian/changelog <<EOF
yandex-id-frontend (${NEXT_VER}) unstable; urgency=low

  [ Fake User ]
  * feat(id): fake commit

 -- $GIT_USER <$DEBEMAIL>  Tue, 06 Oct 2015 11:59:34 +0300

yandex-id-frontend (${PREV_VER}) unstable; urgency=low

  [ Yet Another Fake User ]
  * feat(id): another fake commit

 -- $GIT_USER <$DEBEMAIL>  Tue, 06 Oct 2015 11:59:34 +0300

EOF
    #gbp dch --ignore-branch --since ${PREV_VER} --git-author -U low -N ${NEXT_VER} -D unstable;
    #mv ../../debian/changelog ./debian/;
}

function initCIConfig {
    export GIT_USER=$ROBOT_CI_LOGIN;
    export DEBEMAIL=$ROBOT_CI_EMAIL;
    if [ "$ROBOT_CI_LOGIN" ]; then
        archon get-gpg-secrets;
    fi

}

function initBuildConfig {
  node ./tools/initBuildConfig.js;
}

echo "Building service in ${SERVICE_ROOT}"

[ "$CI" ] && initCIConfig;

initUserName GIT_USER;
initUserEmail DEBEMAIL;
initPgpkey;
initSshkey;

cd "${SERVICE_ROOT}/"; # services/id

downloadIpregDB;
genChangelog;

initBuildConfig;

# Собираем пакеты
debuild --preserve-env --prepend-path=$PATH -b \
    -k"${DEBEMAIL}" \
    -p"gpg2 -s --homedir ${GPGDIR} --pinentry-mode=loopback";

uploadPackages;

# переносим файл пакета для сборки внутри докер-контейнера
mv ../yandex-id-frontend_*_all.deb ./
