import { writeFileSync } from 'fs';

import { RegionType } from '@yandex-int/yandex-geobase';

import { createGeobaseLookup } from './geobase-lookup';

function processTimezoneName(name: string) {
  const cities = name.split(',');

  if (cities.length > 3) {
    return cities.slice(0, 3).join(', ');
  }

  return name;
}
const geobaseOptions = {
  dataPath: process.env.GEOBASE_DATA_PATH || '/var/cache/geobase/geodata6.bin',
};

const lookup = createGeobaseLookup(geobaseOptions);

const langs = ['ru', 'en', 'uk', 'tr', 'fr'];

const countries = lookup.getRegionsByType(RegionType.COUNTRY);

const contriesResult = langs.reduce((acc, lang) => {
  return {
    ...acc,
    [lang]: countries
      .map((country) => ({
        name: lookup.getLinguistics(country.id, lang).nominative,
        code: country.iso_name.toLowerCase(),
      }))
      .filter((country) => country.name !== '')
      .sort((a, b) => {
        if (a.name > b.name) {
          return 1;
        } else if (a.name < b.name) {
          return -1;
        }

        return 0;
      }),
  };
}, {});

const timezones = lookup.getKnownTimezones();
const timezonesResult = langs.reduce((acc, lang) => {
  return {
    ...acc,
    [lang]: timezones
      .filter((timezone) => timezone.name !== '')
      .sort((a, b) => {
        const nameA = a.desc_langs[lang] ? a.desc_langs[lang] : a.desc_langs.en;
        const nameB = b.desc_langs[lang] ? b.desc_langs[lang] : b.desc_langs.en;

        if (nameA > nameB) {
          return 1;
        } else if (nameA < nameB) {
          return -1;
        }

        return 0;
      })
      .map((timezone) => ({
        code: timezone.name,
        name: timezone.desc_langs_with_offset[lang]
          ? processTimezoneName(timezone.desc_langs_with_offset[lang])
          : processTimezoneName(timezone.desc_langs_with_offset.en),
      })),
  };
}, {});

writeFileSync(
  process.cwd() + '/src/server/features/location/countries.json',
  JSON.stringify(contriesResult),
);
writeFileSync(
  process.cwd() + '/src/server/features/location/timezones.json',
  JSON.stringify(timezonesResult),
);
