/* eslint-disable import/no-extraneous-dependencies */
'use strict';

const { getCachedDocumentNodeFromSchema, oldVisit } = require('@graphql-codegen/plugin-helpers');

const header = `
/*
* -------------------------------------------------------
* THIS FILE WAS AUTOMATICALLY GENERATED (DO NOT MODIFY)
* -------------------------------------------------------
*/
`;

module.exports = {
  plugin(schema, _documents, config) {
    const astNode = getCachedDocumentNodeFromSchema(schema);

    const scalarTypes = `
      export type ID = string;
      export type Int = number;
      export type Float = number;
      export type String = string;
      export type Boolean = boolean;
    `;

    const utilityTypes = `
      type Nullable<T> = T | null;
    `;

    const result = oldVisit(astNode, {
      leave: {
        SchemaDefinition() {
          return '';
        },
        ScalarTypeDefinition(node) {
          if (config.scalars[node.name.value]) {
            return createTypeDeclaration(node.name.value, config.scalars[node.name.value]);
          }
        },
        NamedType(node) {
          return wrapWithNullable(node.name.value);
        },
        NonNullType(node) {
          return stripNullable(node.type);
        },
        ListType(node) {
          return wrapWithNullable(`Array<${node.type}>`);
        },
        FieldDefinition(node) {
          const symbol = isNullable(node.type) ? '?:' : ':';

          if (node.name.value === 'query') {
            return `${node.name.value}${symbol} ${wrapWithEmptyQuery(node.type)};`;
          }

          return `${node.name.value}${symbol} ${node.type};`;
        },
        InputValueDefinition(node) {
          const symbol = isNullable(node.type) ? '?:' : ':';

          return `${node.name.value}${symbol} ${node.type};`;
        },
        ObjectTypeDefinition(node) {
          if (node.name.value === 'Query' || node.name.value === 'Mutation') {
            return createClassDeclaration(node.name.value, [], []);
          }
          const interfaces = node.interfaces.map((value) => stripNullable(value));

          return createClassDeclaration(node.name.value, node.fields, interfaces);
        },
        InputObjectTypeDefinition(node) {
          return createClassDeclaration(node.name.value, node.fields, []);
        },
        EnumValueDefinition(node) {
          return `${node.name.value} = "${node.name.value}",`;
        },
        EnumTypeDefinition(node) {
          return createEnumDeclaration(node.name.value, node.values);
        },
        UnionTypeDefinition(node) {
          const types = node.types.map((value) => stripNullable(value));

          return createUnionDeclaration(node.name.value, types);
        },
        InterfaceTypeDefinition(node) {
          return createInterfaceDeclaration(node.name.value, node.fields);
        },
      },
    });

    return {
      prepend: [header, scalarTypes, utilityTypes],
      content: [result.definitions.join('\n')].filter(Boolean).join('\n'),
    };
  },
};

function wrapWithEmptyQuery(value) {
  return `${value} = {}`;
}

function wrapWithNullable(value) {
  return `Nullable<${value}>`;
}

function stripNullable(value) {
  return value.replace(/Nullable<(.*?)>$/, '$1');
}

function isNullable(value) {
  return /^Nullable/.test(value);
}

function createTypeDeclaration(name, type) {
  return `export type ${name} = ${type};`;
}

function createClassDeclaration(name, fields, interfaces) {
  let _interface = '';

  if (interfaces.length > 0) {
    _interface = `implements ${interfaces.join(' | ')}`;
  }

  return `
    export class ${name} ${_interface} {
      __typename?: "${name}" = "${name}";
      ${fields.join('\n')}
    }
  `;
}

function createEnumDeclaration(name, fields) {
  return `
    export enum ${name} {
      ${fields.join('\n')}
    }
  `;
}

function createUnionDeclaration(name, types) {
  return createTypeDeclaration(name, types.join(' | '));
}

function createInterfaceDeclaration(name, fields) {
  return `
    export interface ${name}{
      ${fields.join('\n')}
    }
  `;
}
