/* eslint-disable array-callback-return */
const path = require('path');
const fs = require('fs');
const crypto = require('crypto');

const { readDirRecursive } = require('./read-dir-recursive');

const projectRoot = path.resolve(__dirname, '../../');
const i18nRE = new RegExp(`${projectRoot}/src/.*i18n/[a-z]{2}.ts`);

const pageRE = /private-next-pages/;
const maxNameLength = 50;

/**
 * Ищем к каким пейджам относится модуль
 * Идем вверх по зависимостям, и смотрим в какие чанки попали все родители
 *
 * @param {Object} startModule - анализируемый модуль
 * @returns {Object[]}
 */
function getParentPagesName(startModule) {
  let match;
  let mod = startModule;
  const pagesSet = new Set();

  // идем вверх по зависимостям
  while (mod.issuer && !(mod.rawRequest && (match = pageRE.exec(mod.rawRequest)))) {
    // смотрим в какие чанки попали родительские модули
    Array.from(mod.issuer.chunksIterable)
      // нас интересуют только чанки с пейджами
      .filter((q) => q.name.indexOf('pages/') === 0)
      .forEach((q) => {
        // отрезаем стартовые pages/ чтобы получить только название пейджа
        // всё складируем в Сет, чтобы получить набор уникальных пейджей без дублирования
        pagesSet.add(q.name.substring(6).replace(/\//g, '_'));
      });

    mod = mod.issuer;
  }

  if (!match) {
    console.warn('i18n module without parent page', module);
  }

  // сортируем, чтобы набор был устойчивым/повторимым
  const pages = Array.from(pagesSet).sort();

  return pages || ['global'];
}

function generateManifestFile(manifest, distDir) {
  const chunkDir = 'static/chunks';

  const fullChunksList = readDirRecursive(path.join(distDir, chunkDir));
  const appPage = manifest._app;

  Object.entries(manifest).forEach(([pageName, page]) =>
    Object.entries(page).forEach(([lang, langChunks]) => {
      // app добавляем всегда, т.к. это корневой контейнер страницы
      if (pageName !== '_app') {
        Object.assign(langChunks, appPage[lang]);
      }

      Object.keys(langChunks).forEach((file) => {
        // мы знаем только префикс будущего чанка
        // чтобы найти полное имя файла берем полный список файлов в папке с чанками и ищем файл по префиксу
        const chunkName = fullChunksList.find((filepath) => filepath.indexOf(file) === 0);

        langChunks[chunkName ? `${chunkDir}/${chunkName}` : file] =
          typeof langChunks[file] === 'function' ? langChunks[file]() : langChunks[file];
      });
    }),
  );

  fs.writeFileSync('.next/locales-manifest.json', JSON.stringify(manifest, null, 4));
}

function getChunkName(pages, lang) {
  let fname = pages.join('_');

  if (fname.length > maxNameLength) {
    const newLen = maxNameLength - 10;

    fname =
      fname.substring(0, newLen) +
      crypto
        .createHash('shake256', { outputLength: 10 })
        .update(fname.substring(newLen))
        .digest('hex');
  }

  return fname + '-lang-' + lang;
}

// смысл модификации: собрать все файлы переводов и разложить их по структуре страница -> язык -> необходимые чанки
// потом данные об этой стукруте нужно собрать в файл-манифест, чтобы можно было его использовать для предзагрузки чанков
module.exports.i18nPatchWebpackConfig = function i18nPatchWebpackConfig(config, options) {
  if (options.isServer || options.dev) {
    return;
  }
  const manifest = {};

  process.on('exit', (code) => {
    if (code === 0) {
      generateManifestFile(manifest, options.config.distDir);
    }
  });

  config.optimization.splitChunks.cacheGroups.i18n = {
    test: i18nRE,
    name(module) {
      const rname = module.nameForCondition();
      let fname = path.basename(rname);
      const pages = getParentPagesName(module);
      const lang = fname.substring(0, fname.lastIndexOf('.'));

      fname = getChunkName(pages, lang);

      pages.forEach((pageName) => {
        const page = manifest[pageName] || (manifest[pageName] = {});

        (page[lang] || (page[lang] = {}))[fname] = () => module.id;
      });

      return fname;
    },
    priority: 150,
    enforce: true,
    chunks: 'all',
  };
};
