import React from 'react';
import type { ReactNode } from 'react';
import { withRouter } from 'react-router';
import type { RouteComponentProps } from 'react-router';

import { logError } from '@src/lib/logError';
import type { InitialSsrErrorMeta } from '@src/types';

import Content from './Content';

type Props = {
    children: ReactNode;
    initialError?: InitialSsrErrorMeta;
} & RouteComponentProps;
type State = { error: Error | null; lastPath: string };

class ErrorBoundary extends React.Component<Props, State> {
    static getDerivedStateFromProps(nextProps: Props, prevState: State): Partial<State> | null {
        const pathname = nextProps.location.pathname;

        return pathname === prevState.lastPath
            ? null
            : {
                  error: null,
                  lastPath: pathname,
              };
    }

    static getDerivedStateFromError(error: Error) {
        return { error };
    }

    state = {
        error: this.props.initialError
            ? Object.assign(new Error('initial error'), this.props.initialError)
            : null,
        lastPath: this.props.location.pathname,
    };

    componentDidCatch(error: Error) {
        if (__DEV__) {
            console.error(error);
        }

        logError(error, {
            sourceMethod: 'errorBoundaryCatch',
            source: 'errorBoundary',
        });
    }

    reloadPage() {
        window.location.reload();
    }

    render() {
        const { error } = this.state;
        const { children } = this.props;

        return !error ? children : <Content />;
    }
}

export default withRouter(ErrorBoundary);
