import React from 'react';

import useConstants from '@hooks/useConstants';
import { i18nCommon } from '@translations/common';
import { canUseDom } from '@use-platform/react/libs/dom-utils';
import { classNames } from '@yandex-int/tap-components/helpers';
import { Popup } from '@yandex-lego/components/Popup/desktop/bundle';

import styles from './index.module.css';

const LANGS = ['ru', 'en', 'kk', 'be'];

const Footer: React.FC = () => {
    const {
        baseUrl,
        lang,
        config: { yandexHost },
    } = useConstants();
    const refLang = React.useRef<HTMLDivElement>(null);
    const [visible, setVisible] = React.useState(false);

    const handleClick = React.useCallback(() => {
        setVisible((prevVisible) => !prevVisible);
    }, []);

    const renderLangs = React.useMemo(
        () =>
            LANGS.map((currentLang) => {
                if (currentLang === lang) {
                    return null;
                }

                return (
                    <a
                        key={currentLang}
                        href={`${baseUrl}/setLang?lang=${currentLang}&retpath=${encodeURIComponent(
                            canUseDom ? window.location.href : '',
                        )}`}
                        className={styles.lang_popup}
                    >
                        {currentLang}
                    </a>
                );
            }),
        [lang, baseUrl],
    );

    const supportLinkTld = React.useMemo(() => 'yandex.' + (lang === 'ru' ? 'ru' : 'com'), [lang]);

    return (
        <footer className={styles.container}>
            <div className={styles.content}>
                <div className={styles.left}>
                    <div
                        ref={refLang}
                        className={classNames(styles.link, styles.lang)}
                        onClick={handleClick}
                    >
                        {lang}
                    </div>

                    <a
                        target="_blank"
                        rel="noopener noreferrer"
                        className={styles.link}
                        href={`https://${supportLinkTld}/support/passport/payment-history.html`}
                    >
                        {i18nCommon('footer_help')}
                    </a>
                    <a
                        target="_blank"
                        rel="noopener noreferrer"
                        className={styles.link}
                        href={`https://forms.${yandexHost}/surveys/10027826.b06ee9fe68aa9768bebe9f0b9c23bca02f610bea/`}
                    >
                        {i18nCommon('footer_feedback')}
                    </a>
                </div>
                <div className={styles.right}>
                    {`© 2001-${new Date().getFullYear()}, `}
                    <a
                        className={classNames(styles.link, styles.link_end)}
                        href={`//${yandexHost}`}
                    >
                        {i18nCommon('yandex')}
                    </a>
                </div>
            </div>

            <Popup
                anchor={refLang}
                visible={visible}
                view="default"
                direction="top"
                target="anchor"
                onClose={handleClick}
            >
                {renderLangs}
            </Popup>
        </footer>
    );
};

export default Footer;
