import i18nFactory from '@i18n';
import * as keyset from '@i18n/Navigation.i18n';
import React, { memo, useMemo } from 'react';
import { Switch, Route } from 'react-router-dom';

import { Page } from '@yandex-id/components';
import { SSRProvider } from '@yandex-lego/components/ssr';
import { cnTheme } from '@yandex-lego/components/Theme';
import { theme } from '@yandex-lego/components/Theme/presets/default';
import { ThemeProvider } from '@yandex-id/design-system';
import { PaymentScreen } from '@client/screens/PaymentScreen';
import { getRoutes } from '@src/routes';
import { Constants, RenderProps } from '@src/types';

import ErrorBoundary from './ErrorBoundary';
import Footer from './Footer';
import styles from './index.module.css';
import {
    IconAccount,
    IconPayments,
    IconServices,
    IconFamily,
    IconUGC,
    IconHelp,
    IconSupport,
    IconPasswords,
    IconSecurity,
} from './NavigationIcons';
import Navigator from './Navigator';
import PwaWarning from './PwaWarning';
import useNetStateChange from './useNetStateChange';
import { User } from './User';
import Wrapper from './Wrapper';

import ErrorToastScreen from '../screens/ErrorToastScreen';
import './index.css';

const i18n = i18nFactory(keyset);

const getPasswordUrl = (platform: Constants['platform']): string => {
    if (platform.OSFamily === 'Android' && (platform.BrowserName === 'YandexBrowser' || platform.BrowserName === 'YandexSearch')) {
        return 'personal-data-manager://?screen=passwords';
    }

    if (platform.OSFamily === 'iOS' && (platform.BrowserName === 'YandexBrowser' || platform.BrowserName === 'YandexSearch')) {
        return 'ya-search-app-open://?uri=personaldata%3A%2F%2Fpasswords';
    }

    return '/profile/passwords';
};

const getPageProps = (platform: Constants['platform'], flags: Constants['flags']) => ({
    user: <User />,
    pathname: '/order-history',
    navigationLinks: [
        {
            href: '/profile',
            title: i18n('Аккаунт'),
            icon: <IconAccount />,
            blank: flags.noHeader,
        },
        {
            href: '/order-history',
            title: i18n('Платежи'),
            icon: <IconPayments />,
            blank: flags.noHeader,
        },
        {
            href: '/profile/family',
            title: i18n('Семья'),
            icon: <IconFamily />,
            blank: flags.noHeader,
        },
        flags.isYandexoid && flags.newSecurity && {
            href: '/security',
            title: ('Безопасность'),
            icon: <IconSecurity />,
            blank: flags.noHeader,
        },
        {
            href: flags.isYandexoid && flags.newHelpdesk ? '/helpdesk' : '/support',
            title: i18n('Центр поддержки'),
            icon: <IconSupport />,
            blank: flags.noHeader,
        },
        {
            href: '/profile/services',
            title: i18n('Мои сервисы'),
            icon: <IconServices />,
            blank: flags.noHeader,
        },
        {
            href: 'https://reviews.yandex.ru/ugcpub/cabinet',
            title: i18n('Отзывы и оценки'),
            icon: <IconUGC />,
            blank: flags.noHeader,
        },
    ].filter(Boolean),
    specialNavigationLinks: [
        {
            href: getPasswordUrl(platform),
            title: ('Пароли сайтов'),
            icon: <IconPasswords />,
            blank: flags.noHeader,
        },
    ],
    otherNavigationLinks: [
        {
            href: '/support/passport/',
            title: i18n('Справка'),
            icon: <IconHelp />,
            blank: flags.noHeader,
        },
    ],
});

interface IContentProps {
    lang: string;
    tld: string;
    isLite?: boolean;
    platform: Constants['platform'];
    flags: Constants['flags'];
}

const Content: React.FC<IContentProps> = memo(({ lang, tld, isLite, platform, flags }) => {
    useNetStateChange();

    const pageProps = React.useMemo(() => getPageProps(platform, flags), [platform, flags]);

    if (isLite) {
        return <Navigator />;
    }

    return (
        <>
            {/* @ts-expect-error Ругается на то, что в navigationLinks может быть false, хотя там .filter(Boolean) */}
            <Page {...pageProps} isHeaderHidden={flags.noHeader} lang={lang} tld={tld}>
                <div className={styles.page}>
                    <Navigator />
                    <Footer />
                </div>
            </Page>
        </>
    );
});

Content.displayName = 'Content';

const App = (props: RenderProps) => {
    const { lang, tld, isPWA, baseUrl, isLite, platform, flags } = props.constants;
    const routes = useMemo(() => getRoutes(baseUrl), [baseUrl]);

    return (
        <SSRProvider>
            <ThemeProvider>
                <Wrapper {...props}>
                    <div className={cnTheme(theme)}>
                        <ErrorBoundary initialError={props.initialSsrErrorMeta}>
                            <Switch>
                                <Route path={routes.order} exact>
                                    <PaymentScreen />
                                </Route>

                                <Route path="*">
                                    <Content
                                        lang={lang}
                                        tld={tld}
                                        isLite={isLite}
                                        platform={platform}
                                        flags={flags}
                                    />
                                </Route>
                            </Switch>

                            {isPWA && <PwaWarning />}
                        </ErrorBoundary>
                    </div>
                    <ErrorToastScreen />
                </Wrapper>
            </ThemeProvider>
        </SSRProvider>
    );
};

export default App;
