import React, { ReactNode } from 'react';

import { cn } from '@bem-react/classname';
import { classnames } from '@bem-react/classnames';

import { isRefunded, isShowStatus } from '@client/helpers';
import { formatPrice } from '@client/helpers/number';
import CashbackValue from '@components/CashbackValue';
import Icon from '@components/Icon';
import Image from '@components/Image';
import PlusValue from '@components/PlusValue';
import StatusOrder from '@components/StatusOrder';

import './index.css';
import styles from './index.module.css';

import { isStrictDefined } from '../../../helpers';

type Props = {
    serviceName: string;
    icon?: Nullable<string>;
    description?: Nullable<string>;
    price?: Nullable<Client.Price>;
    currency?: Nullable<string>;
    status?: Nullable<Client.StatusOrder>;
    plus?: Nullable<number>;
    className?: string;
    initiatorAvatar?: string;
    initiatorName?: string;
    isReceipt?: boolean;
    cashback?: number;
};

const cnCard = cn('Card');

const stylesStatus = {
    cancelled: styles.status__cancelled,
    refunded: styles.status__refunded,
};

const Card: React.FC<Props> = ({
    serviceName, icon, description, price, status, plus, currency, initiatorName, initiatorAvatar, isReceipt, cashback
}) => {
    const refunded = isRefunded(status);

    const renderPrice = React.useMemo(() => {
        if (isStrictDefined(price) && price !== 0) {
            return <div className={styles.price}>{formatPrice(price, currency, refunded)}</div>;
        }

        if (plus) {
            return <PlusValue mode="medium" value={plus} refunded={refunded} />;
        }

        return null;
    }, [price, plus, currency, refunded]);

    const renderPriceAdditional = React.useMemo(() => {
        if (isShowStatus(status)) {
            return <StatusOrder status={status} className={classnames(styles.status, stylesStatus[status])} classNameIcon={styles.statusIcon} size="xxs" />;
        }

        let cashbackBadge: ReactNode | undefined;
        let plusBadge: ReactNode | undefined;

        if (cashback && price !== 0) {
            cashbackBadge = <CashbackValue mode="small" value={cashback} />;
        }

        if (plus && isStrictDefined(price) && price !== 0) {
            plusBadge = <PlusValue className={styles.plusValue} mode="small" value={plus} refunded={refunded} />;
        }

        if (cashbackBadge || plusBadge) {
            return (
                <>
                    {cashbackBadge}
                    {plusBadge}
                </>
            );
        }

        return null;
    }, [status, plus, price, refunded, cashback]);

    const renderServiceLogo = React.useCallback((size: number, className?: string) => {
        return icon ?
            <Image cover width={size} height={size} alt={serviceName} className={className} src={icon} /> :
            <Icon cover width={size} height={size} className={className} name={isReceipt ? 'shop' : 'Yandex'} />;
    }, [icon, serviceName, isReceipt]);

    return (
        <div className={styles.card}>
            <div className={styles.logo__block}>
                {initiatorAvatar ? (
                    <>
                        <Image
                            cover
                            width={40}
                            height={40}
                            alt={initiatorName}
                            className={styles.logo}
                            src={initiatorAvatar} />
                        {renderServiceLogo(16, cnCard('Logo', { additional: true }))}
                    </>
                ) : renderServiceLogo(40, cnCard('Logo'))}
            </div>
            <div className={styles.info}>
                <div className={styles.aboutService}>
                    <div className={styles.aboutService__block}>
                        <div className={styles.serviceName}>{[initiatorName, serviceName].filter(Boolean).join(' • ')}</div>
                        {description ? <div className={styles.description}>{description}</div> : null}
                    </div>
                </div>
                <div className={classnames(styles.price__block, refunded ? styles.price__refunded : undefined)}>
                    {renderPrice}
                    <div className={styles.price__description}>{renderPriceAdditional}</div>
                </div>
            </div>
        </div>
    );
};

export default Card;
