import React from 'react';

import { classNames } from '@yandex-int/tap-components/helpers';

import { useIsIE } from '@hooks/useIsIE';

import styles from './index.module.css';

export type Props = Omit<JSX.IntrinsicElements['img'], 'ref' | 'loading' | 'aria-label' | 'aria-hidden'> & {
    cover?: boolean;
    ariaLabel?: string;
    className?: string;
};

const Image = ({ alt, cover, ariaLabel, className, onError, onLoad, ...rest }: Props) => {
    const [hasError, setHasError] = React.useState(false);
    const isIe = useIsIE();

    const handleError = React.useCallback(
        (error) => {
            onError?.(error);
            setHasError(true);
        },
        [onError],
    );

    const handleLoad = React.useCallback(
        (event) => {
            onLoad?.(event);
            setHasError(false);
        },
        [onLoad],
    );

    if (isIe) {
        return (
            <div
                className={classNames(className, styles.image_ie)}
                style={{ backgroundImage: `url(${rest.src})`, width: rest.width, height: rest.height }}
            />
        );
    }

    return (
        <img
            alt={alt}
            aria-label={ariaLabel}
            aria-hidden={!ariaLabel}
            role={ariaLabel && 'img'}
            className={classNames(className, styles.image, cover && styles.image_cover, hasError && styles.image_error)}
            loading="lazy"
            onError={handleError}
            onLoad={handleLoad}
            {...rest}
        />
    );
};

export default Image;
