import React, { ChangeEvent, MouseEvent, useCallback } from 'react';

import { classNames } from '@yandex-int/tap-components/helpers';

import { formatPlus, formatPrice } from '@client/helpers/number';
import Checkbox from '@components/Checkbox';
import Icon from '@components/Icon';

import styles from './index.module.css';

export interface PenaltyCardProps {
    className?: string;
    price: number;
    originalPrice?: number;
    plus?: number;
    currency: string;
    negative?: boolean;
    info: string;
    status?: string;
    checkboxEnabled?: boolean;
    checked?: boolean;
    onCheckedChange?: (event: ChangeEvent<HTMLInputElement>) => void;
}

const PenaltyCard: React.FC<PenaltyCardProps> = ({
    className, price, originalPrice, plus, currency, negative, info, status, checkboxEnabled, checked, onCheckedChange,
}: PenaltyCardProps) => {
    const onCheckboxClick = useCallback((ev: MouseEvent<HTMLElement>) => {
        ev.preventDefault();
        ev.stopPropagation();
    }, []);

    return (
        <div className={classNames(styles['penalty-card'], negative && styles.negative, className)}>
            <div className={styles['first-line']}>
                <span className={styles.price}>{formatPrice(price, currency, false, { signed: false })}</span>
                {originalPrice !== undefined &&
                    <span className={styles['original-price']}>
                        {formatPrice(originalPrice, currency, false, { signed: false })}
                    </span>}
                {plus !== undefined &&
                    <>
                        <Icon className={styles['plus-icon']} name="plus" size="xxs" />
                        <span className={styles['plus-value']}>{formatPlus(plus, false, {})}</span>
                    </>
                }
                {checkboxEnabled &&
                    <Checkbox
                        checked={checked}
                        onChange={onCheckedChange}
                        onClick={onCheckboxClick}
                        className={styles.checkbox}
                        view="outline"
                        size="m"
                        variant="yellow"
                    />
                }
            </div>
            <div className={styles['second-line']}>
                <span>{info}</span>
            </div>
            {status ? (
                <div className={styles['third-line']}>
                    <span className={classNames(styles.status, negative && styles.negative)}>{status}</span>
                </div>
            ) : null}
        </div>
    );
};

export default PenaltyCard;
