import React from 'react';
import { CSSTransition } from 'react-transition-group';

import { classNames } from '@yandex-int/tap-components/helpers';

import styles from './index.module.css';

type Type = 'opacity';
type TypeClassNames = Record<'appearActive' | 'appearDone' | 'enterActive' | 'enterDone', string>;
type Props = {
    appear?: boolean;
    in?: boolean;
    className?: string;
    unmountOnExit?: boolean;
} & (
    | {
        type?: Type;
        activeClassName?: string;
    }
    | {
        type: 'custom';
        activeClassName: string;
    }
);

const getClassNames = (className: string): TypeClassNames => ({
    appearActive: className,
    appearDone: className,
    enterActive: className,
    enterDone: className,
});
const getTypeClassNames = (type: Type, active?: boolean): TypeClassNames =>
    getClassNames(styles[type + (active ? '_active' : '')]);

const TRANSITION_DURATION = 200;
const TRANSITION_CLASSNAMES: Record<Type, TypeClassNames> = {
    opacity: getTypeClassNames('opacity', true),
};

const Transition: React.FC<Props> = ({
    in: _in,
    children,
    className,
    appear = true,
    activeClassName,
    unmountOnExit = true,
    type = 'opacity',
}) => {
    return (
        <CSSTransition
            in={_in ?? true}
            appear={appear}
            unmountOnExit={unmountOnExit}
            classNames={type === 'custom' ? getClassNames(activeClassName || '') : TRANSITION_CLASSNAMES[type]}
            timeout={{
                enter: TRANSITION_DURATION,
                appear: TRANSITION_DURATION,
                exit: TRANSITION_DURATION,
            }}
        >
            <div className={classNames(className, ...(type === 'custom' ? [] : [styles[type], styles.child]))}>
                {children}
            </div>
        </CSSTransition>
    );
};

export default Transition;
