import * as D from 'date-fns';
import type { Locale } from 'date-fns';

import { i18nCommon } from '@client/translations/common';

export type DateFNSDate = Date | number;
export type AutoDateOptions = {
    checkIfToday?: boolean;
    timeFormat?: string | null;
    yearFormat?: string | null;
};

const CURRENT_YEAR = new Date().getFullYear();

export const formatDate = (date: DateFNSDate, formatStr: string, locale: Locale) =>
    D.format(date, formatStr, { locale }).trim();

export const formatDateAuto = (
    date: string,
    { timeFormat, checkIfToday, yearFormat = 'yyyy' }: AutoDateOptions,
    locale: Locale,
): string => {
    const dateObject = new Date(date);

    if (isNaN(dateObject.getTime())) {
        return '';
    }

    const isCurrentYear = new Date(date).getFullYear() === CURRENT_YEAR;

    if (checkIfToday) {
        if (D.isToday(dateObject)) {
            return i18nCommon('today');
        }

        if (D.isYesterday(dateObject)) {
            return i18nCommon('yesterday');
        }
    }

    return (
        formatDate(dateObject, 'd MMMM' + (yearFormat && !isCurrentYear ? ' ' + yearFormat : ''), locale) +
        (timeFormat ? `${D.format(dateObject, timeFormat)}` : '')
    );
};
