const SPACE = ' ';
const NUMBER_SPACE_REGEX = /\B(?=(\d{3})+(?!\d))/g;

export const formatNumber = (num: number | string) => {
    const parts = String(num).replace('.', ',').split(',');

    return parts[0].replace(NUMBER_SPACE_REGEX, SPACE) + (parts[1] ? ',' + parts[1] : '');
};

const MAP_CURRENCY_NAME_TO_SYMBOL: Record<string, string> = {
    AMD: '֏',
    AZN: '₼',
    BYN: 'BYN',
    EUR: '€',
    GEL: '₾',
    ILS: '₪',
    KGS: 'с',
    KZT: '₸',
    MDL: 'L',
    RSD: 'din',
    RUB: '₽',
    RUR: '₽',
    USD: '$',
    UZS: 'сум',
};

export const formatCurrency = (currency: Nullable<string> | undefined): string => {
    return currency ? MAP_CURRENCY_NAME_TO_SYMBOL[currency] || currency : '';
};

interface FormatPriceOptions {
    roundIfNeeded?: boolean;
    signed?: boolean;
}

function doRoundIfNeeded(value: number) {
    return value === Math.round(value) ? value.toFixed(0) : value.toFixed(2);
}

export const formatPrice = (
    price: Client.Price,
    currency?: Nullable<string>,
    refunded: Nullable<boolean> = false,
    { roundIfNeeded = true, signed = true }: FormatPriceOptions = {},
) => {
    const sign = refunded ? '+' : '−';
    const value = roundIfNeeded ? doRoundIfNeeded(Math.abs(price)) : Math.abs(price).toFixed(2);
    const prefix = price === 0 || !signed ? '' : sign;

    return `${prefix}${formatNumber(value)} ${formatCurrency(currency)}`;
};

interface FormatPlusOptions {
    leadPlus?: boolean;
    signed?: boolean;
}

export const formatPlus = (
    plus: number,
    refunded: boolean,
    { leadPlus = false, signed = true }: FormatPlusOptions = {},
) => {
    const amount = refunded ? -plus : plus;
    const leadPlusSign = leadPlus ? '+' : '';
    const sign = amount < 0 ? '−' : leadPlusSign;
    const prefix = signed ? sign : '';

    return `${prefix}${formatNumber(Math.abs(amount))}`;
};
