import { i18nPayments } from '@client/translations/payments';
import { IOrder } from '@src/types';

import { isStrictDefined } from '../../helpers';
import { isPlusPaymentMethod } from '../../helpers/data';

export interface OrderPayment {
    method: Client.PaymentMethod;
    price?: Nullable<Client.Price>;
    account?: Nullable<string>;
    currency?: Nullable<string>;
    refunded?: Nullable<boolean>;
}

interface PaymentCard extends OrderPayment {
    account: string;
}

export interface OrderDetail {
    name: string;
    price: Client.Price;
    currency?: Nullable<string>;
    delivery?: Nullable<boolean>;
    hidden?: Nullable<boolean>;
    alias?: Nullable<string>;
}

export function isPaymentCard(payment: OrderPayment): payment is PaymentCard {
    return payment.method === 'card' && 'account' in payment && isStrictDefined(payment.account);
}

export function isPaymentPlatformPay(payment: OrderPayment): boolean {
    return payment.method === 'google_pay' || payment.method === 'apple_pay';
}

export type PaymentSystem = 'Mir' | 'AmericanExpress' | 'Visa' | 'MasterCard' | 'UnionPay' | 'UnknownCard';
export type PlatformPay = 'google_pay' | 'apple_pay';

const numberToPaymentSystem: Record<string, PaymentSystem> = {
    '2': 'Mir',
    '3': 'AmericanExpress',
    '4': 'Visa',
    '5': 'MasterCard',
    '6': 'UnionPay',
};

const hiddenAliases = ['delivery', 'service_fee'];

export function getPaymentSystemByMask(mask: string = ''): PaymentSystem {
    return numberToPaymentSystem[mask.charAt(0)] || 'UnknownCard';
}

export function getNameByCardAndPaymentSystem(
    mask: string,
    paymentSystem?: PaymentSystem,
): string {
    if (!mask || paymentSystem === 'UnknownCard') {
        return i18nPayments('card');
    }

    return `${i18nPayments(paymentSystem ? paymentSystem : 'card')} •••• ${mask.slice(-4)}`;
}

const knownPaymentMethods = [
    'card',
    'yamoney_wallet',
    'cash',
    'new_promocode',
    'spasibo',
    'yandex_account_withdraw',
    'yandex_account_topup',
    'afisha_certificate',
    'google_pay',
    'apple_pay',
    'virtual_kinopoisk',
    'cashback_edadeal',
];

export const getPaymentMethodKey = (method: Client.PaymentMethod): string => {
    let _method: string = method;

    switch (method) {
        // TODO: проверить правильность названия виртуального платежа
        case 'virtual::kinopoisk_card_discounts':
        case 'virtual::kinopoisk_subs_discounts':
            _method = 'virtual_kinopoisk';

            break;
        case 'virtual::new_promocode':
            _method = 'new_promocode';

            break;
        case 'spasibo_cashback':
            _method = 'spasibo';

            break;
    }

    if (knownPaymentMethods.indexOf(_method) >= 0) {
        return _method;
    }

    return 'paying';
};

export const isPlusTopup = ({ payments, status }: IOrder) => {
    return (
        payments &&
        payments.length &&
        (payments.every((payment) => payment.method === 'yandex_account_topup') ||
            (status === 'refunded' && payments.every((payment) => payment.method === 'yandex_account_withdraw')))
    );
};

export const isPlusTransaction = ({ plus, payments }: IOrder) => {
    return isStrictDefined(plus) && payments && payments.length === 1 && isPlusPaymentMethod(payments[0].method);
};

interface SortingOrder {
    [key: string]: number;
}

export const methodSortingOrder: SortingOrder = {
    card: 1,
    yandex_account_withdraw: 2,
};

export function compareOrderPayments(key: keyof OrderPayment, sortingOrder: SortingOrder) {
    return function (a: OrderPayment, b: OrderPayment) {
        const first = (String(a[key]) in sortingOrder) ? sortingOrder[String(a[key])] : Number.MAX_SAFE_INTEGER;
        const second = (String(b[key]) in sortingOrder) ? sortingOrder[String(b[key])] : Number.MAX_SAFE_INTEGER;

        return first - second;
    };
}

export function isHiddenAlias(alias?: string) {
    return hiddenAliases.includes(alias);
}
