import { useMemo } from 'react';

import { billsPayTokenType, billsPayAuthReturnUrl } from '@client/constants';
import {
    IBillsApiResult,
    IBillPayApiRequest,
    IBillPayApiResult,
    IBillPaymentStatusApiResult,
    IDocument,
    IBillCreateOrderApiResult,
    ClientMpiInfo,
} from '@src/types';

import { useRequest } from './useRequest';

const getClientMpiInfo = (): ClientMpiInfo => {
    return {
        window_height: window.outerHeight,
        window_width: window.outerWidth,
        browser_screen_height: window.screen.height,
        browser_screen_width: window.screen.width,
        browser_color_depth: window.screen.colorDepth,
        browser_tz: String((new Date()).getTimezoneOffset()),
    };
};

export const useBillsApi = () => {
    const request = useRequest({ timeout: 20000 });

    return useMemo(() => {
        async function getAllDocuments(): Promise<IDocument[]> {
            return (await request.get<IDocument[]>('/bills/documents')).data;
        }

        async function createDocument(data: Omit<IDocument, 'id'>): Promise<IDocument> {
            return (await request.post<IDocument>('/bills/documents', data)).data;
        }

        async function updateDocument(id: string, data: Omit<IDocument, 'id'>): Promise<IDocument> {
            return (await request.put<IDocument>(`/bills/documents/${id}`, data)).data;
        }

        async function deleteDocument(id: string): Promise<{}> {
            return (await request.delete<{}>(`/bills/documents/${id}`)).data;
        }

        async function getBills(): Promise<IBillsApiResult> {
            return (await request.get<IBillsApiResult>('/bills/bills')).data;
        }

        async function restartSearch(): Promise<{}> {
            return (await request.post<{}>('/bills/search/bills')).data;
        }

        async function createOrder(bills: string[]): Promise<IBillCreateOrderApiResult> {
            return (await request.post<IBillCreateOrderApiResult>('/bills/orders', { bills })).data;
        }

        async function startOrder(_data: Omit<IBillPayApiRequest, 'mpi_3ds_info' | 'payment_method' | 'return_url'>, orderId: string): Promise<IBillPayApiResult> {
            const data = {
                ..._data,
                mpi_3ds_info: getClientMpiInfo(),
                payment_method: billsPayTokenType,
                return_url: billsPayAuthReturnUrl(),
            } as IBillPayApiRequest;

            return (await request.post<IBillPayApiResult>(`/bills/orders/${orderId}/start`, data)).data;
        }

        async function getTransaction(transactionId: string): Promise<IBillPaymentStatusApiResult> {
            return (await request.get<IBillPaymentStatusApiResult>(`/bills/transactions/${transactionId}`)).data;
        }

        async function setNotificationsAgreement(agree: boolean): Promise<{}> {
            return (await request.post<IBillCreateOrderApiResult>('/bills/notifications', { agree })).data;
        }

        return {
            getAllDocuments,
            createDocument,
            updateDocument,
            deleteDocument,
            getBills,
            restartSearch,
            createOrder,
            startOrder,
            getTransaction,
            setNotificationsAgreement,
        };
    }, [request]);
};
