import { useCallback, useEffect, useRef, useState, useMemo } from 'react';

import {
    resetPayment,
    startPayment,
    startPaymentAuth,
    completePaymentAuth,
    // waitTokenPayment,
    processPayment,
    failPayment,
    successPayment,
} from '@client/redux/bills';
import { Order, Payment, PaymentData } from '@client/types/yandex-pay';
import { loadYandexPayScript } from '@lib/yandex-pay';
import { IBillPaymentToken, IBillPayApiResult, IBillPaymentAuth3ds, IBill } from '@src/types';

import { useDispatch, useSelector } from './redux';
import { useBillsApi } from './useBillsApi';
import useConstants from './useConstants';

type PaymentDataWithoutOrder = Omit<PaymentData, 'order'>;

const PAYMENT_STATUS_POOL_INTERVAL = 5 * 1000;

function getAuth3dsData(data: IBillPayApiResult): null | IBillPaymentAuth3ds {
    const url3ds = data.transaction.acs_url;

    if (!url3ds) {
        return null;
    }

    return {
        method: 'post',
        url: url3ds,
    };
}

const getInitialPaymentData = (): PaymentDataWithoutOrder => {
    const { YaPay } = window;

    // Сформировать данные платежа.
    return {
        env: YaPay.PaymentEnv.Testing,
        version: 2,
        countryCode: YaPay.CountryCode.Ru,
        currencyCode: YaPay.CurrencyCode.Rub,
        merchant: {
            id: '18b3b6ac-6d17-41ff-ad6a-cbed3775c3e3',
            name: 'passport-bills',
        },
        paymentMethods: [
            {
                type: YaPay.PaymentMethodType.Card,
                gateway: /* 'mobimoney' */ 'yandex-trust',
                gatewayMerchantId: 'oplatagosuslug',
                allowedAuthMethods: [YaPay.AllowedAuthMethod.PanOnly],
                allowedCardNetworks: [
                    YaPay.AllowedCardNetwork.Visa,
                    YaPay.AllowedCardNetwork.Mastercard,
                    YaPay.AllowedCardNetwork.Mir,
                    YaPay.AllowedCardNetwork.Maestro,
                    YaPay.AllowedCardNetwork.VisaElectron,
                ],
            },
        ],
        requiredFields: {
            billingContact: { email: true, name: true },
        },
    };
};

const getPrice = (amount: number) => (amount / 100).toFixed(2);

const getOrderByBills = (bills: IBill[], orderId: string) => {
    const sum = bills.reduce((total, bill) => total + bill.discounted_amount, 0);
    const fee = sum * 0.05;

    return {
        id: orderId,
        total: { label: 'Итого', amount: getPrice(sum + fee) },
        items: [
            { label: 'Сумма штрафа', amount: getPrice(sum) },
            { label: 'Комиссия', amount: getPrice(fee) },
        ],
    };
};

export const useBillsPayment = () => {
    const dispatch = useDispatch();

    const billsPayment = useSelector((state) => state.bills.payment);
    const { createOrder, startOrder, getTransaction } = useBillsApi();

    const [paymentData, setPaymentData] = useState<Nullable<PaymentDataWithoutOrder>>(null);
    const [paymentDataOrder, setPaymentDataOrder] = useState<Nullable<Order>>(null);
    const [billIds, setBillIds] = useState<string[]>([]);
    const [orderId, setOrderId] = useState<string>('');
    const [payment, setPayment] = useState<Nullable<Payment>>(null);
    const { yandexPay: yandexPayConfig } = useConstants();

    useEffect(() => {
        loadYandexPayScript(yandexPayConfig.scriptSrc, () =>
            setPaymentData(getInitialPaymentData()),
        );
    }, []);

    const pollStatusTimer = useRef<NodeJS.Timeout>();

    useEffect(() => {
        const stopPool = () => {
            if (pollStatusTimer.current) {
                clearTimeout(pollStatusTimer.current);
                pollStatusTimer.current = undefined;
            }
        };

        const poolPayment = () => {
            if (billsPayment.transactionId) {
                getTransaction(billsPayment.transactionId).then((data) => {
                    if (data.transaction.status === 'NEW') {
                        pollStatusTimer.current = setTimeout(
                            poolPayment,
                            PAYMENT_STATUS_POOL_INTERVAL,
                        );
                    } else if (data.transaction.status === 'PAID') {
                        dispatch(successPayment());
                    } else {
                        dispatch(failPayment());
                    }
                });
            } else {
                stopPool();
            }
        };

        if (billsPayment.transactionId) {
            poolPayment();
        }

        return stopPool;
    }, [dispatch, getTransaction, billsPayment.transactionId]);

    const setBills = useCallback(
        (bills: IBill[]) => {
            const ids = bills.map((bill) => bill.bill_id);

            setBillIds(ids);

            createOrder(ids).then((data) => {
                setOrderId(data.order.order_id);
                setPaymentDataOrder(getOrderByBills(bills, data.order.order_id));
            });
        },
        [createOrder],
    );

    const showYandexPay = useCallback(
        (node: HTMLElement) => {
            if (!payment) {
                return;
            }

            const { YaPay } = window;

            const button = payment.createButton({
                type: YaPay.ButtonType.Pay,
                theme: YaPay.ButtonTheme.Yellow,
                width: YaPay.ButtonWidth.Max,
            });

            button.mount(node);

            button.on(YaPay.ButtonEventType.Click, function onPaymentButtonClick() {
                payment.checkout();
            });
        },
        [payment],
    );

    const startPaymentFn = useCallback(
        (paymentToken: IBillPaymentToken, userName: string) => {
            dispatch(startPayment({ token: paymentToken }));

            const payData = {
                payment_token: paymentToken,
                payer_full_name: userName,
            };

            startOrder(payData, orderId).then(
                (data) => {
                    const auth3ds = getAuth3dsData(data);

                    if (auth3ds) {
                        dispatch(startPaymentAuth(data.transaction.transaction_id, auth3ds));
                    } else {
                        dispatch(processPayment(data.transaction.transaction_id));
                    }
                },
                () => {
                    dispatch(failPayment());
                },
            );
        },
        [dispatch, startOrder, orderId],
    );

    const completePaymentAuthFn = useCallback(() => {
        dispatch(completePaymentAuth());
    }, [dispatch]);

    const resetPaymentFn = useCallback(() => {
        dispatch(resetPayment());
    }, [dispatch]);

    useEffect(() => {
        if (!paymentDataOrder || !paymentData) {
            return;
        }

        if (!payment) {
            const { YaPay } = window;

            YaPay.createPayment({
                ...paymentData,
                order: paymentDataOrder,
            }).then((_payment) => {
                setPayment(_payment);

                _payment.on(YaPay.PaymentEventType.Error, function onPaymentError() {
                    _payment.complete(YaPay.CompleteReason.Error, 'error');
                });

                _payment.on(YaPay.PaymentEventType.Abort, function onPaymentAbort() {
                    // TODO
                });

                _payment.on(YaPay.PaymentEventType.Process, function onPaymentProcess(event) {
                    startPaymentFn(event.token, event.billingContact?.name || 'Иванов Иван');

                    _payment.complete(YaPay.CompleteReason.Success);
                });
            });
        } else {
            payment.update({ order: paymentDataOrder });
        }
    }, [paymentData, paymentDataOrder, payment, billIds, startPaymentFn]);

    const payReady = useMemo(
        () => Boolean(payment && paymentDataOrder && paymentData),
        [payment, paymentData, paymentDataOrder],
    );

    return {
        showYandexPay,
        startPayment: startPaymentFn,
        completePaymentAuth: completePaymentAuthFn,
        resetPayment: resetPaymentFn,
        setBills,
        payReady,
    };
};
