import { useCallback } from 'react';

import {
    startPayment,
    // waitTokenPayment,
    cancelPayment,
    processPayment,
    startPaymentAuth,
    completePaymentAuth,
    failPayment,
    resetPayment,
} from '@client/redux/bills';
import {
    IBillPaymentToken,
    IBillPayApiResult,
    IBillPaymentAuth3ds,
} from '@src/types';

import { useDispatch } from './redux';
import { useBillsApi } from './useBillsApi';

function getAuth3dsData(data: IBillPayApiResult): null | IBillPaymentAuth3ds {
    const url3ds = data.transaction.acs_url;

    if (!url3ds) {
        return null;
    }

    return {
        method: 'post',
        url: url3ds,
    };
}

export const useBillsPayment = () => {
    const dispatch = useDispatch();

    const { startOrder } = useBillsApi();

    const startPaymentFn = useCallback(
        (orderId: string, paymentToken: IBillPaymentToken, userName: string) => {
            dispatch(startPayment({ token: paymentToken }));

            const payData = {
                payment_token: paymentToken,
                payer_full_name: userName,
            };

            startOrder(payData, orderId).then(
                (data) => {
                    const auth3ds = getAuth3dsData(data);

                    if (auth3ds) {
                        dispatch(startPaymentAuth(data.transaction.transaction_id, auth3ds));
                    } else {
                        dispatch(processPayment(data.transaction.transaction_id));
                    }
                },
                () => {
                    dispatch(failPayment('UNABLE_TO_START_ORDER'));
                },
            );
        },
        [dispatch, startOrder],
    );

    const completePaymentAuthFn = useCallback(
        () => {
            dispatch(completePaymentAuth());
        },
        [dispatch],
    );

    const failPaymentFn = useCallback(
        (reason?: string) => {
            dispatch(failPayment(reason));
        },
        [dispatch],
    );

    const cancelPaymentFn = useCallback(
        () => {
            dispatch(cancelPayment());
        },
        [dispatch],
    );

    const resetPaymentFn = useCallback(
        () => {
            dispatch(resetPayment());
        },
        [dispatch],
    );

    return {
        startPayment: startPaymentFn,
        completePaymentAuth: completePaymentAuthFn,
        failPayment: failPaymentFn,
        cancelPayment: cancelPaymentFn,
        resetPayment: resetPaymentFn,
    };
};
