import { useEffect, useMemo, useRef, useState } from 'react';

import { addDocuments, setBillsState, setDocumentsState, setSelectedBills, updateBills } from '@client/redux/bills';

import { useDispatch, useSelector } from './redux';
import { useBillsApi } from './useBillsApi';
import useConstants from './useConstants';

const HARD_WORK_EMULATION_TIMEOUT = 2 * 1000;

const BILLS_POLLING_INTERVAL = 5 * 1000;
const BILLS_POLLING_LIMIT = 9;

export const useBillsPolling = () => {
    const { getAllDocuments, getBills } = useBillsApi();
    const dispatch = useDispatch();

    const documents = useSelector((state) => state.bills.documents);
    const bills = useSelector((state) => state.bills.bills);

    const [pollsCount, setPollsCount] = useState(0);

    const { bills: { maxSelected } } = useConstants();

    const timerRef = useRef<ReturnType<typeof setTimeout>>();
    const pollsCountRef = useRef(pollsCount);

    pollsCountRef.current = pollsCount;

    useEffect(
        () => {
            if (documents.state === 'initial') {
                getAllDocuments().then((documents) => {
                    dispatch(addDocuments(documents));
                    dispatch(setDocumentsState('stable'));
                });
            }
        },
        [documents.state, getAllDocuments, dispatch],
    );

    useEffect(
        () => {
            if (bills.state === 'initial') {
                getBills().then((bills) => {
                    dispatch(updateBills({
                        state: bills.state === 'completed' ? 'emulation' : bills.state,
                        list: bills.bills,
                    }));
                });
            }
        },
        [bills.state, getBills, dispatch],
    );

    useEffect(
        () => {
            if (bills.state === 'emulation') {
                setTimeout(
                    () => {
                        dispatch(setBillsState('completed'));
                    },
                    HARD_WORK_EMULATION_TIMEOUT,
                );
            }
        },
        [bills.state, dispatch],
    );

    useEffect(
        () => {
            if (bills.state === 'completed') {
                const copy = Array.from(bills.list);

                copy.sort((a, b) => Date.parse(a.payment_deadline) - Date.parse(b.payment_deadline));

                dispatch(setSelectedBills(copy.map((bill) => bill.bill_id).slice(0, maxSelected)));
            }
        },
        [bills.state, bills.list, dispatch, maxSelected],
    );

    useEffect(
        () => {
            if (bills.state === 'outdated') {
                if (timerRef.current) {
                    clearTimeout(timerRef.current);
                    timerRef.current = undefined;
                }

                if (documents.state === 'stable' && documents.list.length > 0) {
                    getBills().then((bills) => {
                        setPollsCount(1);
                        dispatch(updateBills({
                            state: bills.state === 'completed' ? 'emulation' : bills.state,
                            list: bills.bills,
                        }));
                    });
                }
            }
        },
        [bills.state, documents.state, documents.list],
    );

    useEffect(
        () => {
            if (bills.state === 'syncing') {
                const stopPolling = () => {
                    if (timerRef.current) {
                        clearTimeout(timerRef.current);
                        timerRef.current = undefined;
                    }
                };

                stopPolling();

                const playPolling = () => {
                    if (pollsCountRef.current + 1 < BILLS_POLLING_LIMIT) {
                        timerRef.current = setTimeout(() => {
                            getBills().then((bills) => {
                                dispatch(updateBills({
                                    state: bills.state,
                                    list: bills.bills,
                                }));
                                setPollsCount(pollsCountRef.current + 1);
                            });
                            playPolling();
                        }, BILLS_POLLING_INTERVAL);
                    }
                };

                playPolling();

                return stopPolling;
            }
        },
        [bills.state],
    );

    const exceeded = useMemo(() => pollsCount >= BILLS_POLLING_LIMIT, [pollsCount]);

    return { exceeded };
};
