import { useMemo } from 'react';
import * as uuid from 'uuid';

import { WebAmPlatform } from '@src/server/middleware/webAm';

import useConstants from './useConstants';

declare global {
    interface Window {
        nativeAMAndroid?: NativeAM;
        webkit?: {
            messageHandlers?: {
                nativeAM?: {
                    postMessage?(message: object, targetOrigin: string): void;
                };
            };
        };
    }
}

type RequestID = string;
type Message = string;
type Version = number; // Integral version of web-AM API
type Data = unknown; // JSON-compatible object.

interface NativeAmMessage {
  requestId: RequestID;
  message: Message;
  version: Version;
  data: Data;
}

interface NativeAM {
    send(message: NativeAmMessage | string): void;
}

interface SetPopupSizePayload {
    mode: 'bottom' | 'top' | 'center' | 'fullscreen';
    corner_radius?: number;
    horizontal_margins?: number; // отступы по краям карточки по горизонтали от границ экрана
    vertical_margins?: number; // отступы по краям карточки по вертикали от границ экрана
    height?: number;
}

class NativeAMImpl implements NativeAM {
    private readonly platform: WebAmPlatform;

    constructor(platform: WebAmPlatform) {
        this.platform = platform;
    }

    send(message: NativeAmMessage) {
        if (this.platform === WebAmPlatform.ANDROID && window.nativeAMAndroid) {
            window.nativeAMAndroid.send(JSON.stringify(message));
        } else if (this.platform === WebAmPlatform.IOS && window.webkit?.messageHandlers?.nativeAM?.postMessage) {
            // Вторым параметром "targetOrigin" надо отдать "*", иначе сообщения не отправляются
            // либо какой-то определенный, если он не будет меняться
            // https://developer.mozilla.org/ru/docs/Web/API/Window/postMessage
            window.webkit.messageHandlers.nativeAM.postMessage(message, '*');
        }
    }

    setPopupSize(payload: SetPopupSizePayload) {
        this.send({
            requestId: uuid.v4(),
            message: 'setPopupSize',
            version: 0,
            data: payload,
        });
    }

    ready() {
        this.send({
            requestId: uuid.v4(),
            message: 'ready',
            version: 0,
            data: {
                status: 'ok',
            },
        });
    }
}

export const useWebAm = () => {
    const { webAmParams } = useConstants();

    return useMemo(() => {
        if (!webAmParams) {
            return;
        }

        return new NativeAMImpl(webAmParams?.app_platform);
    }, []);
};
