import { useCallback, useEffect, useMemo, useRef, useState } from 'react';

import { parse } from 'query-string';

import { getBillsSum, getFee, getPrice } from '@client/helpers/bills';
import { waitPay } from '@client/redux/bills';
import { Button, Order, Payment, PaymentData, PaymentEnv, YaPay } from '@client/types/yandex-pay';
import { useDispatch } from '@hooks/redux';
import useConstants from '@hooks/useConstants';
import { loadYandexPayScript } from '@lib/yandex-pay';
import { Constants, IBill } from '@src/types';

import { useHistory } from 'react-router';
import { useBillsApi } from './useBillsApi';
import { useBillsPayment } from './useBillsPayment';
import useMetrika from './useMetrika';

type PaymentDataWithoutOrder = Omit<PaymentData, 'order'>;

const getInitialPaymentData = (config: Constants['yandexPay']): PaymentDataWithoutOrder => {
    const { YaPay } = window;

    // Сформировать данные платежа.
    return {
        env: config.env as PaymentEnv,
        version: 2,
        countryCode: YaPay.CountryCode.Ru,
        currencyCode: YaPay.CurrencyCode.Rub,
        merchant: {
            id: config.merchantId,
            name: config.merchantName,
        },
        paymentMethods: [
            {
                type: YaPay.PaymentMethodType.Card,
                gateway: config.gateway,
                gatewayMerchantId: config.gatewayId,
                allowedAuthMethods: [YaPay.AllowedAuthMethod.PanOnly],
                allowedCardNetworks: [
                    YaPay.AllowedCardNetwork.Visa,
                    YaPay.AllowedCardNetwork.Mastercard,
                    YaPay.AllowedCardNetwork.Mir,
                    YaPay.AllowedCardNetwork.Maestro,
                    YaPay.AllowedCardNetwork.VisaElectron,
                ],
            },
        ],
        requiredFields: {
            billingContact: { email: true, name: true },
        },
    };
};

const getOrderByBills = (bills: IBill[], orderId: string): PaymentData['order'] => {
    const sum = getBillsSum(bills);
    const fee = getFee(bills);

    return {
        id: orderId,
        total: { label: 'Итого', amount: getPrice(sum + fee) },
        items: [
            { label: 'Сумма штрафа', amount: getPrice(sum) },
            { label: 'Комиссия', amount: getPrice(fee) },
        ],
    };
};

type PayRef = {
    init?: boolean;
    payment?: Payment;
    button?: Button;
    orderId?: string;
};

export const useYandexPay = () => {
    const metrika = useMetrika();
    const dispatch = useDispatch();

    const history = useHistory();

    const { createOrder } = useBillsApi();
    const { startPayment, failPayment, cancelPayment } = useBillsPayment();
    const { yandexPay: yandexPayConfig } = useConstants();

    // Ref сделан для проброса данных в removeYandexPay и yaPay.PaymentEventType.Process
    const [yaPay, setYaPay] = useState<YaPay>();
    const [payment, setPayment] = useState<Payment>();
    const [button, setButton] = useState<Button>();
    const [buttonNode, setButtonNode] = useState<HTMLElement>();
    const [paymentOrder, setPaymentOrder] = useState<Nullable<Order>>(null);

    const pay = useRef<PayRef>({});

    useEffect(() => {
        loadYandexPayScript(yandexPayConfig.scriptSrc, () => setYaPay(window.YaPay));
    }, [yandexPayConfig.scriptSrc]);

    const updatePay = useCallback((update: Partial<PayRef>) => {
        pay.current = { ...pay.current, ...update };
    }, []);

    useEffect(() => {
        const { init, payment } = pay.current;

        if (yaPay && paymentOrder && !init && !payment) {
            updatePay({ init: true });

            yaPay.createPayment({
                ...getInitialPaymentData(yandexPayConfig),
                order: paymentOrder,
            })
                .then((_payment) => {
                    setPayment(_payment);
                    updatePay({ payment: _payment });

                    _payment.on(yaPay.PaymentEventType.Error, function onPaymentError() {
                        if (_payment.sheet.order.id === pay.current.orderId) {
                            failPayment();

                            window.localStorage.removeItem('order-id');

                            _payment.complete(yaPay.CompleteReason.Error, 'error');
                        }
                    });

                    _payment.on(yaPay.PaymentEventType.Abort, function onPaymentAbort() {
                        if (_payment.sheet.order.id === pay.current.orderId) {
                            cancelPayment();
                            
                            window.localStorage.removeItem('order-id');

                            _payment.complete(yaPay.CompleteReason.Close);
                        }
                    });

                    _payment.on(yaPay.PaymentEventType.Process, function onPaymentProcess(event) {
                        if (_payment.sheet.order.id === pay.current.orderId) {
                            startPayment(pay.current.orderId, event.token, event.billingContact?.name || 'Иванов Иван');

                            window.localStorage.removeItem('order-id');

                            _payment.complete(yaPay.CompleteReason.Success);
                        }
                    });
                });
        }
    }, [yaPay, paymentOrder, updatePay, cancelPayment, failPayment, startPayment, yandexPayConfig]);

    useEffect(() => {
        const { payment, orderId } = pay.current;

        if (yaPay && payment && paymentOrder && paymentOrder.id !== orderId) {
            payment.update({ order: paymentOrder });
        }
    }, [yaPay, paymentOrder, updatePay]);

    useEffect(() => {
        if (yaPay && buttonNode && payment) {
            const button = payment.createButton({
                type: yaPay.ButtonType.Pay,
                theme: yaPay.ButtonTheme.Yellow,
                width: yaPay.ButtonWidth.Max,
            });

            button.mount(buttonNode);

            button.on(yaPay.ButtonEventType.Click, function onPaymentButtonClick() {
                metrika.reachGoal('click_pay', {});

                const { sheet: {order: {id: orderId}}} = payment;
                
                window.localStorage.setItem('order-id', orderId);

                payment.checkout();
                updatePay({ orderId });
                dispatch(waitPay());
            });

            setButton(button);
            updatePay({ button });
        }
    }, [yaPay, buttonNode, updatePay, payment, dispatch, metrika]);

    const isYandexPayReady = useMemo(() => Boolean(button), [button]);

    const updateYandexPay = useCallback((node: HTMLElement, bills: IBill[]) => {
        setButtonNode(node);

        const localOrderId = window.localStorage.getItem('order-id');
        const qs = parse(history.location.search);

        // Если пришли с редиректа Pay, значит ожидаем оплату и транзакция в статусе Process.
        if (localOrderId && qs.__YP__) {
            setPaymentOrder(getOrderByBills([], localOrderId));
            updatePay({ orderId: localOrderId });
            dispatch(waitPay());
        } else if (bills.length) {
            const ids = bills.map((bill) => bill.bill_id);

            createOrder(ids).then((data) => { 
                setPaymentOrder(getOrderByBills(bills, data.order.order_id));
            });
        }
    }, [createOrder, history.location.search, dispatch, updatePay]);

    const removeYandexPay = useCallback(() => {
        const { payment, button } = pay.current;

        if (button) {
            button.unmount();
        }

        if (payment) {
            payment.destroy();
        }
    }, []);

    return {
        isYandexPayReady,
        updateYandexPay,
        removeYandexPay,
    };
};
