import { Reducer } from 'redux';

import {
    BillsState,
    IBillStateType,
    IDocument,
    IDocumentStateType,
    IBillPaymentId,
    IBillPaymentToken,
    IBillPaymentStateCreate,
    IBillPaymentStateProcess,
    IBillPaymentStateAuth3ds,
    IBillPaymentAuth3ds,
    IBillStateWithoutSelected,
} from '@src/types';

type PaymentCreatePayload = {
    token: IBillPaymentToken;
};

type PaymentProcessPayload = {
    transactionId: IBillPaymentId;
};

type PaymentStartAuthPayload = {
    transactionId: IBillPaymentId;
    auth3ds: IBillPaymentAuth3ds;
};

export type BillsAction =
    | { type: 'BILLS_ADD_DOCUMENT'; payload: IDocument }
    | { type: 'BILLS_ADD_DOCUMENTS'; payload: IDocument[] }
    | { type: 'BILLS_UPDATE_DOCUMENT'; payload: IDocument }
    | { type: 'BILLS_REMOVE_DOCUMENT'; payload: string }
    | { type: 'BILLS_SET_DOCUMENTS_STATE'; payload: IDocumentStateType }
    | { type: 'BILLS_UPDATE_BILLS'; payload: IBillStateWithoutSelected }
    | { type: 'BILLS_SET_SELECTED_BILLS'; payload: string[] }
    | { type: 'BILLS_SET_BILLS_STATE'; payload: IBillStateType }
    | { type: 'BILLS_PAYMENT_CREATE'; payload: PaymentCreatePayload }
    | { type: 'BILLS_PAYMENT_WAIT_TOKEN' }
    | { type: 'BILLS_PAYMENT_CANCEL' }
    | { type: 'BILLS_PAYMENT_PROCESS'; payload: PaymentProcessPayload }
    | { type: 'BILLS_PAYMENT_AUTH_START'; payload: PaymentStartAuthPayload }
    | { type: 'BILLS_PAYMENT_AUTH_COMPETE' }
    | { type: 'BILLS_PAYMENT_FAIL'; payload?: string }
    | { type: 'BILLS_PAYMENT_SUCCESS' }
    | { type: 'BILLS_PAYMENT_RESET' }
    | { type: 'BILLS_PAYMENT_WAIT_PAY' };

export const addDocuments = (documents: IDocument[]) =>
    ({ type: 'BILLS_ADD_DOCUMENTS', payload: documents } as BillsAction);
export const addDocument = (document: IDocument) =>
    ({ type: 'BILLS_ADD_DOCUMENT', payload: document } as BillsAction);
export const updateDocument = (document: IDocument) =>
    ({ type: 'BILLS_UPDATE_DOCUMENT', payload: document } as BillsAction);
export const removeDocument = (id: string) =>
    ({ type: 'BILLS_REMOVE_DOCUMENT', payload: id } as BillsAction);
export const setDocumentsState = (state: IDocumentStateType) =>
    ({ type: 'BILLS_SET_DOCUMENTS_STATE', payload: state } as BillsAction);
export const updateBills = (state: IBillStateWithoutSelected) =>
    ({ type: 'BILLS_UPDATE_BILLS', payload: state } as BillsAction);
export const setBillsState = (state: IBillStateType) =>
    ({ type: 'BILLS_SET_BILLS_STATE', payload: state } as BillsAction);
export const setSelectedBills = (state: string[]) =>
    ({ type: 'BILLS_SET_SELECTED_BILLS', payload: state } as BillsAction);

export const waitPay = () => ({ type: 'BILLS_PAYMENT_WAIT_PAY' } as BillsAction);
export const waitTokenPayment = () => ({ type: 'BILLS_PAYMENT_WAIT_TOKEN' } as BillsAction);
export const startPayment = (state: PaymentCreatePayload) =>
    ({ type: 'BILLS_PAYMENT_CREATE', payload: state } as BillsAction);
export const cancelPayment = () => ({ type: 'BILLS_PAYMENT_CANCEL' } as BillsAction);
export const processPayment = (transactionId: string) =>
    ({ type: 'BILLS_PAYMENT_PROCESS', payload: { transactionId } } as BillsAction);
export const startPaymentAuth = (transactionId: string, auth3ds: IBillPaymentAuth3ds) =>
    ({ type: 'BILLS_PAYMENT_AUTH_START', payload: { transactionId, auth3ds } } as BillsAction);
export const completePaymentAuth = () => ({ type: 'BILLS_PAYMENT_AUTH_COMPETE' } as BillsAction);
export const failPayment = (reason?: string) =>
    ({ type: 'BILLS_PAYMENT_FAIL', payload: reason } as BillsAction);
export const successPayment = () => ({ type: 'BILLS_PAYMENT_SUCCESS' } as BillsAction);
export const resetPayment = () => ({ type: 'BILLS_PAYMENT_RESET' } as BillsAction);

export const DEFAULT_BILLS_STATE: BillsState = {
    documents: {
        state: 'initial',
        list: [],
    },
    bills: {
        state: 'initial',
        list: [],
        selected: [],
    },
    payment: {
        state: 'initial',
        transactionId: '',
    },
};

export const billsReducer: Reducer<BillsState, BillsAction> = (
    state: BillsState | undefined,
    action: BillsAction,
) => {
    // eslint-disable-next-line no-param-reassign
    state = state ?? DEFAULT_BILLS_STATE;

    switch (action.type) {
        case 'BILLS_ADD_DOCUMENTS': {
            return {
                ...state,
                documents: {
                    ...state.documents,
                    list: [...(state.documents.list ?? []), ...action.payload],
                },
            };
        }
        case 'BILLS_ADD_DOCUMENT': {
            return {
                ...state,
                documents: {
                    ...state.documents,
                    list: [...(state.documents.list ?? []), action.payload],
                },
            };
        }
        case 'BILLS_UPDATE_DOCUMENT': {
            return {
                ...state,
                documents: {
                    ...state.documents,
                    list: state.documents.list.map((doc) =>
                        doc.id === (action.payload as IDocument).id ? action.payload : doc,
                    ) ?? [action.payload],
                },
            };
        }
        case 'BILLS_REMOVE_DOCUMENT': {
            return {
                ...state,
                documents: {
                    ...state.documents,
                    list: state.documents.list.filter(
                        (doc) => doc.id !== (action.payload as string),
                    ),
                },
            };
        }
        case 'BILLS_SET_DOCUMENTS_STATE': {
            return {
                ...state,
                documents: {
                    ...state.documents,
                    state: action.payload,
                },
            };
        }
        case 'BILLS_UPDATE_BILLS': {
            return {
                ...state,
                bills: {
                    ...state.bills,
                    ...action.payload,
                },
            };
        }
        case 'BILLS_SET_BILLS_STATE': {
            return {
                ...state,
                bills: {
                    ...state.bills,
                    state: action.payload,
                },
            };
        }
        case 'BILLS_SET_SELECTED_BILLS': {
            return {
                ...state,
                bills: {
                    ...state.bills,
                    selected: action.payload,
                },
            };
        }

        // BILL_PAYMENT
        case 'BILLS_PAYMENT_WAIT_PAY': {
            return {
                ...state,
                payment: {
                    state: 'wait-pay',
                    transactionId: '',
                },
            };
        }
        case 'BILLS_PAYMENT_WAIT_TOKEN': {
            return {
                ...state,
                payment: {
                    state: 'wait-token',
                    transactionId: '',
                },
            };
        }
        case 'BILLS_PAYMENT_CREATE': {
            return {
                ...state,
                payment: {
                    state: 'create',
                    transactionId: '',
                    token: action.payload.token,
                },
            };
        }
        case 'BILLS_PAYMENT_PROCESS': {
            return {
                ...state,
                payment: {
                    ...(state.payment as IBillPaymentStateCreate),
                    state: 'process',
                    transactionId: action.payload.transactionId,
                },
            };
        }
        case 'BILLS_PAYMENT_AUTH_START': {
            return {
                ...state,
                payment: {
                    ...(state.payment as IBillPaymentStateProcess),
                    state: 'auth-3ds',
                    transactionId: action.payload.transactionId,
                    auth3ds: action.payload.auth3ds,
                },
            };
        }
        case 'BILLS_PAYMENT_AUTH_COMPETE': {
            return {
                ...state,
                payment: {
                    ...(state.payment as IBillPaymentStateAuth3ds),
                    state: 'process',
                },
            };
        }
        case 'BILLS_PAYMENT_CANCEL': {
            return {
                ...state,
                payment: {
                    ...(state.payment as IBillPaymentStateCreate),
                    state: 'cancel',
                },
            };
        }
        case 'BILLS_PAYMENT_FAIL': {
            return {
                ...state,
                payment: {
                    ...(state.payment as IBillPaymentStateProcess),
                    state: 'fail',
                    reason: action.payload,
                },
            };
        }
        case 'BILLS_PAYMENT_SUCCESS': {
            return {
                ...state,
                payment: {
                    ...(state.payment as IBillPaymentStateProcess),
                    state: 'success',
                },
            };
        }
        case 'BILLS_PAYMENT_RESET': {
            return {
                ...state,
                payment: {
                    state: 'initial',
                    transactionId: '',
                },
            };
        }

        default:
            return state;
    }
};
