import i18nFactory from '@i18n';
import * as keyset from '@i18n/Gibdd.i18n';
import React from 'react';

import { cn } from '@bem-react/classname';

import './index.css';
import { compareOrderPayments, methodSortingOrder, OrderPayment } from '@client/helpers/order';
import Payment from '@client/screens/MainScreen/OrdersDetails/OrderContent/Payment';
import { useDateAutoFormatter } from '@hooks/useDateFormatters';

import PenaltyNegativeAbout from '../Negative';

interface PenaltyDetail {
    name: string;
    content: string;
}

interface Receipt {
    details: PenaltyDetail[];
    by: string;
}

interface Props {
    negative?: boolean;
    paidOn?: string;
    payments?: OrderPayment[];
    refunded?: boolean;
    resolution: string;
    details?: PenaltyDetail[];
    requisites?: PenaltyDetail[];
    receipt?: Receipt;
}

const cnPenaltyContent = cn('PenaltyContent');

const i18n = i18nFactory(keyset);

const PenaltyDetailsContent: React.FC<Props> = ({
    negative,
    paidOn,
    payments,
    refunded = false,
    resolution,
    details,
    requisites,
    receipt,
}) => {
    const formatDate = useDateAutoFormatter();

    const datetime = React.useMemo(() => {
        return paidOn ? formatDate(paidOn, { timeFormat: ', HH:mm' }) : null;
    }, [paidOn, formatDate]);

    const paymentMethods = React.useMemo(() => {
        return payments && payments.length ? (
            <div>
                {[...payments]
                    .sort(compareOrderPayments('method', methodSortingOrder))
                    .map((payment, index) => (
                        <Payment data={payment} refunded={refunded} key={`${payment.method}_${index}`} />
                    ))
                }
            </div>
        ) : null;
    }, [payments, refunded]);

    return (
        <div className={cnPenaltyContent()}>
            {negative && <PenaltyNegativeAbout />}
            {paidOn && <span className={cnPenaltyContent('dateline')}>{datetime}</span>}
            {paymentMethods && <div className={cnPenaltyContent('payments')}>{paymentMethods}</div>}
            {paidOn && payments && payments.length > 0 && <div className={cnPenaltyContent('divider')} />}
            <h1 className={cnPenaltyContent('header')}>{i18n('Детали штрафа')}</h1>
            <span className={cnPenaltyContent('info')}>{resolution}</span>
            {details && details.map((detail) => (
                <React.Fragment key={detail.name}>
                    <h2 className={cnPenaltyContent('header')}>{detail.name}</h2>
                    <span className={cnPenaltyContent('info')}>{detail.content}</span>
                </React.Fragment>
            ))}
            {requisites &&
                <div className={cnPenaltyContent('requisites')}>
                    {requisites.map((requisite) => (
                        <div key={requisite.name} className={cnPenaltyContent('requisit-line')}>
                            <span className={cnPenaltyContent('requisit-name')}>{requisite.name}</span>
                            <span className={cnPenaltyContent('requisit-value')}>{requisite.content}</span>
                        </div>
                    ))}
                </div>
            }
            {receipt &&
                <React.Fragment>
                    <div className={cnPenaltyContent('divider')} />
                    {receipt?.details.map((detail) => (
                        <div key={detail.name} className={cnPenaltyContent('receipt-line')}>
                            <span className={cnPenaltyContent('receipt-name')}>{detail.name}</span>
                            <span className={cnPenaltyContent('receipt-value')}>{detail.content}</span>
                        </div>
                    ))}
                    <div className={cnPenaltyContent('receipt-by')}>{receipt.by}</div>
                </React.Fragment>
            }
        </div>
    );
};

export default PenaltyDetailsContent;
