import i18nFactory from '@i18n';
import * as keyset from '@i18n/Gibdd.i18n';
import React, { useCallback, useMemo } from 'react';

import { isExpired } from '@client/helpers/bills';
import { formatPrice } from '@client/helpers/number';
import { BillPaymentYaPay } from '@client/screens/GibddScreen/Payment/YandexPay';
import { getNormalizedDocName } from '@client/screens/GibddScreenSettings/GibddScreenSettings.utils';
import DetailLayout from '@components/DetailLayout';
import { useSelector } from '@hooks/redux';
import useConstants from '@hooks/useConstants';
import { useDateFormatter } from '@hooks/useDateFormatters';
import { IBill } from '@src/types';

import PenaltyDetailsContent from './Content';
import PenaltyDetailsHeader from './Header';

export type PenaltyDetailsProps = {
    bill: IBill;
}

const i18n = i18nFactory(keyset);

const PenaltyDetails: React.FC<PenaltyDetailsProps> = ({ bill }) => {
    const { payServices } = useConstants();
    const dateFormatter = useDateFormatter();
    const documents = useSelector((state) => state.bills.documents.list);
    const gibddService = useMemo(() =>
        payServices.filter((payService) => payService.path === '/order-history/gibdd')[0], [payServices]);

    const isNegative = useMemo(() => isExpired(bill), [bill]);

    const status = useMemo(
        () => {
            if (!isNegative) {
                if (!bill.payment_deadline) {
                    return '';
                }

                return `${dateFormatter(Date.parse(bill.payment_deadline), 'dd.MM.yyyy')}`;
            }

            return i18n('Просрочен');
        },
        [bill.payment_deadline, isNegative, dateFormatter],
    );

    // Валюта
    const price = useMemo(
        () => formatPrice(bill.discounted_amount / 100, 'RUB', false, { signed: false }),
        [bill.discounted_amount],
    );

    const documentDescription = useMemo(
        () => {
            const doc = documents.filter((document) => document.id === bill.document_id)[0];

            if (doc) {
                return `${getNormalizedDocName(doc.type)} ${doc.value}`;
            }

            return '';
        },
        [bill, documents],
    );

    // i18n и документ
    const resolution = useMemo(
        () => [
            `${i18n('Постановление №')}${bill.supplier_bill_id}`,
            `${i18n('от')} ${dateFormatter(Date.parse(bill.bill_date), 'dd.MM.yyyy')}`,
            `${i18n('выписан по документу')} ${documentDescription}`,
        ].join(' '),
        [bill.supplier_bill_id, bill.bill_date, dateFormatter, documentDescription],
    );

    const details = useMemo(
        () => {
            const arr = [];

            if (bill.purpose || bill.legal_act) {
                if (bill.purpose && bill.legal_act) {
                    arr.push({
                        name: i18n('Назначение платежа'),
                        content: `${bill.purpose}, ${bill.legal_act}`,
                    });
                } else if (bill.purpose) {
                    arr.push({
                        name: i18n('Назначение платежа'),
                        content: `${bill.purpose}`,
                    });
                } else {
                    arr.push({
                        name: i18n('Назначение платежа'),
                        content: `${bill.legal_act}`,
                    });
                }
            }

            if (bill.offense_place) {
                arr.push({
                    name: i18n('Место нарушения'),
                    content: `${bill.offense_place}`,
                });
            }

            if (bill.payer_name) {
                arr.push({
                    name: i18n('Плательщик'),
                    content: `${bill.payer_name}`,
                });
            }

            if (bill.payee_name) {
                arr.push({
                    name: i18n('Получатель платежа'),
                    content: `${bill.payee_name}`,
                });
            }

            return arr;
        },
        [bill],
    );

    const requisites = useMemo(
        () => {
            const arr = [];

            if (bill.account_number) {
                arr.push({
                    name: i18n('Р/счёт'),
                    content: `${bill.account_number}`,
                });
            }

            if (bill.inn) {
                arr.push({
                    name: i18n('ИНН'),
                    content: `${bill.inn}`,
                });
            }

            if (bill.kpp) {
                arr.push({
                    name: i18n('КПП'),
                    content: `${bill.kpp}`,
                });
            }

            if (bill.bik) {
                arr.push({
                    name: i18n('БИК'),
                    content: `${bill.bik}`,
                });
            }

            if (bill.oktmo) {
                arr.push({
                    name: i18n('ОКТМО'),
                    content: `${bill.oktmo}`,
                });
            }

            return arr;
        },
        [bill],
    );

    const renderHeader = useCallback(() => {
        return <PenaltyDetailsHeader
            negative={isNegative}
            splashUrl={gibddService.splashUrl}
            iconUrl={gibddService.detailsIconUrl}
            number={bill.supplier_bill_id}
            amount={price}
            due={status}
        />;
    }, [gibddService, isNegative, bill.supplier_bill_id, price, status]);

    const renderContent = useCallback(() => {
        return <PenaltyDetailsContent
            negative={isNegative}
            resolution={resolution}
            details={details}
            requisites={requisites}
        />;
    }, [isNegative, resolution, details, requisites]);

    const renderFooter = useCallback(() => {
        return <BillPaymentYaPay bills={[bill]} />;
    }, [bill]);

    return (
        <DetailLayout
            renderHeader={renderHeader}
            renderInfo={renderContent}
            renderFooter={renderFooter}
        />
    );
};

export default PenaltyDetails;
