import i18nFactory from '@i18n';
import * as keyset from '@i18n/Gibdd.i18n';
import React, { useCallback, useMemo } from 'react';
import { useDispatch } from 'react-redux';
import { Link } from 'react-router-dom';

import { cn } from '@bem-react/classname';

import { isExpired } from '@client/helpers/bills';
import { setSelectedBills } from '@client/redux/bills';
import { getNormalizedDocName } from '@client/screens/GibddScreenSettings/GibddScreenSettings.utils';
import PenaltyCard from '@components/PenaltyCard';
import { useSelector } from '@hooks/redux';
import { useDateFormatter } from '@hooks/useDateFormatters';
import useRoutes from '@hooks/useRoutes';
import { IBill, IDocument } from '@src/types';

import { GibddScreenListBanner } from './Banner/Banner';
import GibddScreenListHeader from './Header';
import { GibddScreenListNoBills } from './NoBills/NoBills';

import { PenaltyDetailsProps } from '../Details/PenaltyDetails';
import { BillPaymentYaPay } from '../Payment/YandexPay';

import './index.css';

export type GibddScreenListProps = {
    documents: IDocument[];
    bills: IBill[];
}

const cnGibddScreenList = cn('GibddScreenList');
const i18n = i18nFactory(keyset);

const GibddScreenList: React.FC<GibddScreenListProps> = ({ bills, documents }: GibddScreenListProps) => {
    const routes = useRoutes();
    const dispatch = useDispatch();
    const dateFormatter = useDateFormatter();
    const { selected } = useSelector((state) => state.bills.bills);

    const sorted = useMemo(() => {
        const copy = Array.from(bills);

        copy.sort((a, b) => Date.parse(a.payment_deadline) - Date.parse(b.payment_deadline));

        return copy;
    }, [bills]);

    const onChange = useCallback((key: string) => {
        return () => {
            if (selected.includes(key)) {
                dispatch(setSelectedBills(selected.filter((k) => k !== key)));
            } else {
                dispatch(setSelectedBills([...selected, key]));
            }
        };
    }, [selected, dispatch]);

    const to = useCallback(
        (bill: IBill) => {
            return (currentLocation: Location) =>
                routes.penalty<PenaltyDetailsProps>(currentLocation, { bill });
        },
        [routes],
    );

    const status = useCallback(
        (bill: IBill) => {
            if (!isExpired(bill)) {
                if (!bill.payment_deadline) {
                    return '';
                }

                return `${i18n('До')} ${dateFormatter(Date.parse(bill.payment_deadline), 'dd.MM.yyyy')}`; // Прикрутить i18n к фразе "До"
            }

            return i18n('Просрочен');
        },
        [dateFormatter],
    );

    const documentDescription = useCallback(
        (bill: IBill) => {
            const doc = documents.filter((document) => document.id === bill.document_id)[0];

            if (doc) {
                return `${getNormalizedDocName(doc.type)} ${doc.value}`;
            }

            return '';
        },
        [documents],
    );

    const hasBills = bills.length > 0;
    const hasSts = useMemo(() => documents.filter((doc) => doc.type === 'VEHICLE_REGISTRATION_CERTIFICATE').length > 0, [documents]);
    const hasDl = useMemo(() => documents.filter((doc) => doc.type === 'DRIVER_LICENSE').length > 0, [documents]);
    const hasBoth = useMemo(() => hasSts && hasDl, [hasSts, hasDl]);

    const header = useMemo(
        () => {
            if (sorted.length === 0) {
                return i18n('Штрафов не найдено, так держать!');
            }

            return i18n('Найдено {count} штрафов', { count: sorted.length });
        },
        [sorted],
    );

    const selectedBills = useMemo(
        () => bills.filter((bill) => selected.includes(bill.bill_id)),
        [selected, bills],
    );

    return (
        <>
            <GibddScreenListHeader header={header} hideHeader={!hasBills} />
            {sorted.map((bill) => (
                <Link className={cnGibddScreenList('Link')} key={bill.bill_id} to={to(bill)}>
                    <PenaltyCard
                        className={cnGibddScreenList('PenaltyList')}
                        price={bill.discounted_amount / 100}
                        originalPrice={bill.amount === bill.discounted_amount ? undefined : bill.amount / 100}
                        currency="RUB"
                        info={documentDescription(bill)}
                        status={status(bill)}
                        checked={selected.includes(bill.bill_id)}
                        onCheckedChange={onChange(bill.bill_id)}
                        negative={isExpired(bill)}
                        checkboxEnabled />
                </Link>
            ))}
            {!hasBills && <GibddScreenListNoBills header={header} />}
            {!hasBoth && hasSts && <GibddScreenListBanner marginType={hasBills ? 'long' : 'short'} type="VEHICLE_REGISTRATION_CERTIFICATE" />}
            {!hasBoth && hasDl && <GibddScreenListBanner marginType={hasBills ? 'long' : 'short'} type="DRIVER_LICENSE" />}

            {hasBills && <BillPaymentYaPay sticky showInfo bills={selectedBills} />}
        </>
    );
};

export default GibddScreenList;
