import i18nFactory from '@i18n';
import * as keyset from '@i18n/Gibdd.i18n';
import React, { useCallback, useState, ChangeEvent, useMemo, FormEvent } from 'react';

import { cn } from '@bem-react/classname';

import { addDocuments, setBillsState } from '@client/redux/bills';
import { getMaskByDocType, getPatternByDocType, getPlaceholderByDocType } from '@client/screens/GibddScreenSettings/GibddScreenSettings.utils';
import Button from '@components/Button';
import Checkbox from '@components/Checkbox';
import PayPlusPromo from '@components/PayPlusPromo';
import Textinput from '@components/Textinput';
import { useDispatch } from '@hooks/redux';
import { useBillsApi } from '@hooks/useBillsApi';
import './index.css';
import useConstants from '@hooks/useConstants';
import useMetrika from '@hooks/useMetrika';
import { useErrors } from '@lib/errors';

const cnAddDocuments = cn('AddDocuments');

const i18n = i18nFactory(keyset);

const LEGAL_MOBILE_AGREEMENT_LINK = 'https://yandex.ru/legal/search_mobile_agreement/';
const LEGAL_RULES_LINK = 'https://yandex.ru/legal/rules/';

const AddDocuments: React.FC = () => {
    const metrika = useMetrika();
    const { addError } = useErrors();
    const dispatch = useDispatch();
    const { platform: { isTouch, BrowserName } } = useConstants();

    const billsApi = useBillsApi();

    const [driverLicense, setDriverLicense] = useState<string>('');
    const [dlError, setDlError] = useState('');
    const [vehicleRegistrationCertificate, setVehicleRegistrationCertificate] = useState<string>('');
    const [vrcError, setVrcError] = useState('');
    const [notificationAgreement, setNotificationAgreement] = useState<boolean>(true);
    const [pending, setPending] = useState<boolean>(false);

    const dlState = useMemo(() => dlError === '' ? undefined : 'error', [dlError]);
    const vrcState = useMemo(() => vrcError === '' ? undefined : 'error', [vrcError]);
    const dlHint = useMemo(() => dlError === '' ? undefined : dlError, [dlError]);
    const vrcHint = useMemo(() => vrcError === '' ? undefined : vrcError, [vrcError]);

    const isPP = useMemo(() => BrowserName === 'YandexSearch', [BrowserName]);

    const validate = useCallback(
        () => {
            setDlError('');
            setVrcError('');

            if (driverLicense === '' && vehicleRegistrationCertificate === '') {
                setDlError(i18n('Необходимо указать номер документа'));
                setVrcError(i18n('Необходимо указать номер документа'));

                return false;
            }

            let valid = true;

            if (driverLicense !== '' && !driverLicense.match(getPatternByDocType('DRIVER_LICENSE'))) {
                setDlError(i18n('Неверный формат водительского удостоверения'));
                valid = false;
            }

            if (vehicleRegistrationCertificate !== '' &&
                !vehicleRegistrationCertificate.match(getPatternByDocType('VEHICLE_REGISTRATION_CERTIFICATE'))) {
                setVrcError(i18n('Неверный формат СТС'));
                valid = false;
            }

            return valid;
        },
        [driverLicense, vehicleRegistrationCertificate, setDlError, setVrcError],
    );

    const onDriverLicenseChange = useCallback((event: ChangeEvent<HTMLInputElement>) => {
        setDriverLicense(event.target.value);
    }, []);

    const onVehicleRegistrationCertificateChange = useCallback((event: ChangeEvent<HTMLInputElement>) => {
        setVehicleRegistrationCertificate(event.target.value);
    }, []);

    const onNotificationAgreementChange = useCallback(() => {
        setNotificationAgreement(!notificationAgreement);
    }, [notificationAgreement]);

    const uploadData = useCallback(
        () => {
            if (!validate()) {
                return;
            }

            metrika.reachGoal('click_search_fines', {});

            setPending(true);

            const promises = [];

            if (driverLicense !== '') {
                promises.push(billsApi.createDocument({
                    type: 'DRIVER_LICENSE',
                    value: driverLicense.replace(/ /g, ''),
                    title: i18n('ВУ'),
                }));
            }

            if (vehicleRegistrationCertificate !== '') {
                promises.push(billsApi.createDocument({
                    type: 'VEHICLE_REGISTRATION_CERTIFICATE',
                    value: vehicleRegistrationCertificate.replace(/ /g, ''),
                    title: i18n('СТС'),
                }));
            }

            billsApi.setNotificationsAgreement(notificationAgreement);

            Promise.all(promises).then((documents) => {
                dispatch(addDocuments(documents));
                dispatch(setBillsState('outdated'));
                setPending(false);
            }).catch((error) => {
                addError({
                    refetch: uploadData,
                    error,
                });
            });
        },
        [
            driverLicense,
            vehicleRegistrationCertificate,
            notificationAgreement,
            validate,
            dispatch,
            billsApi,
            addError,
            metrika,
        ],
    );

    const onFormSubmit = useCallback((event: FormEvent<HTMLFormElement>) => {
        event.preventDefault();

        uploadData();
    }, [uploadData]);

    return (
        <form onSubmit={onFormSubmit} className={cnAddDocuments({ touch: isTouch })}>
            <PayPlusPromo />

            <Textinput
                size="m"
                view="material"
                label={i18n('Номер СТС')}
                disabled={pending}
                state={vrcState}
                hint={vrcHint}
                placeholder={getPlaceholderByDocType('VEHICLE_REGISTRATION_CERTIFICATE')}
                className={cnAddDocuments('vehicleRegistrationCertificate')}
                value={vehicleRegistrationCertificate}
                onChange={onVehicleRegistrationCertificateChange}
                mask={getMaskByDocType('VEHICLE_REGISTRATION_CERTIFICATE')}
            />

            <Textinput
                size="m"
                view="material"
                label={i18n('Номер ВУ')}
                disabled={pending}
                state={dlState}
                hint={dlHint}
                placeholder={getPlaceholderByDocType('DRIVER_LICENSE')}
                className={cnAddDocuments('driverLicense')}
                value={driverLicense}
                onChange={onDriverLicenseChange}
                mask={getMaskByDocType('DRIVER_LICENSE')}
            />

            {/* // TODO: добавить в отправку на сервер */}
            <Checkbox
                disabled={pending}
                checked={notificationAgreement}
                onChange={onNotificationAgreementChange}
                label={i18n('Получать уведомления о новых штрафах')}
                className={cnAddDocuments('notificationAgreement')}
                view="outline"
                size="m"
                variant="yellow"
            />

            <Button
                type="submit"
                size="l"
                view="action"
                width="max"
                className={cnAddDocuments('toSearch')}
                disabled={pending}
                progress={pending}
            >
                {i18n('Искать штрафы')}
            </Button>

            <div className={cnAddDocuments('privacy')}>
                {i18n('Нажимая на кнопку, я соглашаюсь с условиями')}&nbsp;
                {isPP ? (
                    <a
                        target="_blank"
                        rel="noopener noreferrer"
                        className={cnAddDocuments('link')}
                        href={LEGAL_MOBILE_AGREEMENT_LINK}
                    >
                        {i18n('Лицензионного соглашения (п.5.10)')}
                    </a>
                ) : (
                    <a
                        target="_blank"
                        rel="noopener noreferrer"
                        className={cnAddDocuments('link')}
                        href={LEGAL_RULES_LINK}
                    >
                        {i18n('Пользовательского соглашения (п.2.8.4)')}
                    </a>
                )}
            </div>
        </form>
    );
};

export default AddDocuments;
