import i18nFactory from '@i18n';
import * as keyset from '@i18n/Gibdd.i18n';
import React, { useEffect, useMemo, useRef } from 'react';

import { parse } from 'query-string';

import { cn } from '@bem-react/classname';

import { getBillsSum, getFee } from '@client/helpers/bills';
import { formatPrice } from '@client/helpers/number';
import Spin from '@components/Spin';
import useConstants from '@hooks/useConstants';
import { useYandexPay } from '@hooks/useYandexPay';
import { IBill } from '@src/types';

import { useHistory } from 'react-router';
import './index.css';

const cnBillPaymentYaPay = cn('BillPaymentYaPay');

const i18n = i18nFactory(keyset);

interface BillPaymentYaPayProps {
    showInfo?: boolean;
    sticky?: boolean;
    bills: IBill[];
}

export const BillPaymentYaPay: React.FC<BillPaymentYaPayProps> = ({
    sticky,
    bills,
}) => {
    const history = useHistory();
    const { isYandexPayReady, updateYandexPay, removeYandexPay } = useYandexPay();
    const { platform: { isTouch }, bills: { maxSelected } } = useConstants();
    const buttonRef = useRef<HTMLDivElement>(null);

    const hasBills = useMemo(() => bills.length > 0, [bills]);
    const billsOverflow = useMemo(() => bills.length > maxSelected, [bills, maxSelected]);
    const buttonReady = isYandexPayReady && hasBills && !billsOverflow;

    const amount = useMemo<number>(() => getBillsSum(bills), [bills]);
    const fee = useMemo<number>(() => getFee(bills), [bills]);
    const amountWithFee = useMemo<number>(() => hasBills ? amount + fee : amount, [amount, fee, hasBills]);
    const maxBillsText = useMemo(() => i18n('Выберите не больше {count} штрафов', { count: maxSelected }), [maxSelected]);

    useEffect(() => {
        if (buttonRef.current) {
            const qs = parse(history.location.search);

            // Здесь произойдёт оплата, если пришли с редиректа Pay
            if (window.localStorage.getItem('order-id') && qs.__YP__) {
                updateYandexPay(buttonRef.current, []);
            } else if (!billsOverflow && bills) {
                updateYandexPay(buttonRef.current, bills);
            }
        }
    }, [updateYandexPay, bills, billsOverflow, buttonRef, history.location.search]);

    useEffect(() => removeYandexPay, [removeYandexPay]);

    return (
        <div className={cnBillPaymentYaPay({ sticky, desktop: !isTouch, touch: isTouch })}>
            <div className={cnBillPaymentYaPay('TextContainer', { desktop: !isTouch, touch: isTouch })}>
                <span className={cnBillPaymentYaPay('TextPrice')}>{i18n('Итого')} {formatPrice(amountWithFee / 100, 'RUB', false, { signed: false })}</span>
                {hasBills ? (
                    <React.Fragment>
                        <span className={cnBillPaymentYaPay('TextDot')}> • </span>
                        <span className={cnBillPaymentYaPay('TextFee')}>{i18n('включая комиссию')} {formatPrice(fee / 100, 'RUB', false, { signed: false })}</span>
                    </React.Fragment>
                ) : null}
            </div>

            <div className={cnBillPaymentYaPay('ButtonContainer')}>
                <div className={cnBillPaymentYaPay('ButtonStatus', { visible: !buttonReady })}>
                    {!hasBills ? (
                        <span className={cnBillPaymentYaPay('NoBillsText')}>{i18n('Выберите счета для оплаты')}</span>
                    ) : null}
                    {billsOverflow ? (
                        <span className={cnBillPaymentYaPay('NoBillsText')}>{maxBillsText}</span>
                    ) : null}
                    {hasBills && !billsOverflow && !isYandexPayReady ? (
                        <Spin size="s" progress />
                    ) : null}
                </div>
                <div ref={buttonRef} className={cnBillPaymentYaPay('Button', { visible: buttonReady })} />
            </div>
        </div>
    );
};
