import i18nFactory from '@i18n';
import * as keyset from '@i18n/Gibdd.i18n';
import React, { ChangeEvent, FormEvent, useCallback, useEffect, useMemo, useState } from 'react';
import { useHistory } from 'react-router';
import { Link } from 'react-router-dom';

import { cn } from '@bem-react/classname';

import { addDocument, removeDocument, setBillsState, updateDocument } from '@client/redux/bills';
import Button from '@components/Button';
import Icon from '@components/Icon';
import IconButton from '@components/IconButton';
import Textinput from '@components/Textinput';
import { useDispatch, useSelector } from '@hooks/redux';
import { useBillsApi } from '@hooks/useBillsApi';
import useMetrika from '@hooks/useMetrika';
import useRoutes from '@hooks/useRoutes';
import { useErrors } from '@lib/errors';
import { IDocumentType } from '@src/types';

import {
    getNormalizedDocName,
    getPatternByDocType,
    getPlaceholderByDocType,
    getMaskByDocType,
    getNormalizeDocumentValue,
} from '../GibddScreenSettings.utils';

import './DocumentForm.css';

const cnDocumentForm = cn('DocumentForm');

export type DocumentFormProps = {
    id?: string;
    addDocumentType?: IDocumentType;
};

const i18n = i18nFactory(keyset);

export const DocumentForm: React.FC<DocumentFormProps> = ({ id, addDocumentType }) => {
    const metrika = useMetrika();
    const routes = useRoutes();
    const documents = useSelector((state) => state.bills.documents.list);
    const billsApi = useBillsApi();
    const history = useHistory();
    const dispatch = useDispatch();
    const { addError } = useErrors();

    const [inProgress, setInProgress] = useState(false);
    const [error, setError] = useState('');
    const [value, setValue] = useState('');

    const state = useMemo(() => (error === '' ? undefined : 'error'), [error]);
    const hint = useMemo(() => (error === '' ? undefined : error), [error]);

    const documentToEdit = useMemo(() => {
        if (id) {
            return documents.filter((document) => document.id === id)[0];
        }

        return undefined;
    }, [documents, id]);

    useEffect(() => {
        if (documentToEdit) {
            setValue(getNormalizeDocumentValue(documentToEdit));
        }
    }, [documentToEdit]);

    const type = useMemo(() => {
        if (documentToEdit) {
            return documentToEdit.type;
        }

        if (addDocumentType) {
            return addDocumentType;
        }

        return undefined;
    }, [documentToEdit, addDocumentType]);

    const normalizedType = useMemo(() => {
        if (type) {
            return getNormalizedDocName(type);
        }

        return '';
    }, [type]);

    const validate = useCallback(() => {
        setError('');

        if (value === '') {
            setError(`${i18n('Необходимо указать номер')} ${normalizedType}`);

            return false;
        }

        if (type && !value.match(getPatternByDocType(type))) {
            setError(`${i18n('Неверный формат номера')} ${normalizedType}`);

            return false;
        }

        return true;
    }, [value, type, normalizedType]);

    const header = useMemo(() => {
        if (documentToEdit) {
            return `${i18n('Редактировать')} ${normalizedType}`;
        }

        if (addDocumentType) {
            return `${i18n('Добавить')} ${normalizedType}`;
        }

        return '';
    }, [documentToEdit, addDocumentType, normalizedType]);

    const onChangeInput = useCallback((e: ChangeEvent<HTMLInputElement>) => {
        setValue(e.target.value);
    }, []);

    const requestChanges = useCallback(() => {
        if (!validate()) {
            return;
        }

        setInProgress(true);
        if (documentToEdit) {
            billsApi
                .updateDocument(documentToEdit.id, {
                    type: documentToEdit.type,
                    title: documentToEdit.title,
                    value: value.replace(/ /g, ''),
                })
                .then((document) => {
                    dispatch(updateDocument(document));
                    dispatch(setBillsState('outdated'));
                    setInProgress(false);
                    // @ts-expect-error
                    history.goBack();
                })
                .catch((error) => addError({ refetch: requestChanges, error }));
        } else {
            const goal = type === 'DRIVER_LICENSE' ? "click_driver's_license" : 'click_CTC';

            metrika.reachGoal(goal, {});

            billsApi
                .createDocument({
                    type: type as IDocumentType,
                    title: normalizedType,
                    value: value.replace(/ /g, ''),
                })
                .then((document) => {
                    dispatch(addDocument(document));
                    dispatch(setBillsState('outdated'));
                    setInProgress(false);
                    // @ts-expect-error
                    history.goBack();
                })
                .catch((error) => addError({ refetch: requestChanges, error }));
        }
    }, [
        value,
        setInProgress,
        addError,
        billsApi,
        dispatch,
        validate,
        documentToEdit,
        normalizedType,
        type,
        history,
        metrika,
    ]);

    const onDelete = useCallback(() => {
        if (documentToEdit && !inProgress) {
            billsApi
                .deleteDocument(documentToEdit.id)
                .then(() => {
                    dispatch(removeDocument(documentToEdit.id));
                    dispatch(setBillsState('outdated'));
                    // @ts-expect-error
                    history.goBack();
                })
                .catch((error) => addError({ refetch: onDelete, error }))
                .finally(() => setInProgress(false));
        }
    }, [addError, billsApi, dispatch, documentToEdit, history, inProgress]);

    const onSubmit = useCallback(
        (event: FormEvent<HTMLFormElement>) => {
            event.preventDefault();

            requestChanges();
        },
        [requestChanges],
    );

    const showForm = useMemo(
        () => addDocumentType || documentToEdit,
        [addDocumentType, documentToEdit],
    );

    const onLinkClick = useCallback<React.MouseEventHandler<HTMLAnchorElement>>(
        (event) => {
            if (inProgress) {
                event.preventDefault();
            }
        },
        [inProgress],
    );

    return (
        <>
            {showForm && (
                <form className={cnDocumentForm()} onSubmit={onSubmit}>
                    <div className={cnDocumentForm('header')}>
                        <Link to={routes.gibddSettings()} onClick={onLinkClick}>
                            <IconButton type="filled">
                                <Icon size="s" name="arrowleft" />
                            </IconButton>
                        </Link>
                    </div>
                    <h1 className={cnDocumentForm('Header')}>{header}</h1>
                    {type && (
                        <Textinput
                            size="m"
                            view="material"
                            label={`Номер ${normalizedType}`}
                            placeholder={getPlaceholderByDocType(type)}
                            state={state}
                            hint={hint}
                            className={cnDocumentForm('Input')}
                            value={value}
                            onChange={onChangeInput}
                            mask={getMaskByDocType(type)}
                        />
                    )}
                    <div className={cnDocumentForm('Footer')}>
                        <Button
                            type="submit"
                            className={cnDocumentForm('AddOrSave')}
                            disabled={inProgress}
                            progress={inProgress}
                            view="action"
                            size="l"
                        >
                            {`${documentToEdit ? i18n('Сохранить') : i18n('Добавить')} ${i18n(
                                'и искать штрафы',
                            )}`}
                        </Button>
                        {documentToEdit && (
                            <Button
                                className={cnDocumentForm('Delete', { inProgress })}
                                view="clear"
                                onClick={onDelete}
                            >
                                {i18n('Удалить')}
                            </Button>
                        )}
                    </div>
                </form>
            )}
        </>
    );
};
