import React from 'react';

import { formatPrice } from '@client/helpers/number';
import {
    getNameByCardAndPaymentSystem,
    getPaymentMethodKey,
    getPaymentSystemByMask,
    isPaymentCard,
    isPaymentPlatformPay,
    OrderPayment,
    PaymentSystem,
    PlatformPay,
} from '@client/helpers/order';
import { i18nPayments, PaymentsKeyType } from '@client/translations/payments';
import CashbackValue from '@components/CashbackValue';
import Icon from '@components/Icon';
import PlusValue from '@components/PlusValue';

import styles from './index.module.css';

import { isStrictDefined } from '../../../../../helpers';

interface Props {
    data: OrderPayment;
    refunded: boolean;
}

const Payment: React.FC<Props> = ({ data, refunded }) => {
    const { name, icon } = React.useMemo<{ name: string; icon?: PaymentSystem | PlatformPay; }>(() => {
        if (isPaymentCard(data)) {
            const paymentSystem = getPaymentSystemByMask(data.account);
            const name = getNameByCardAndPaymentSystem(data.account, paymentSystem);

            return {
                name,
                icon: paymentSystem,
            };
        }

        const name = i18nPayments(getPaymentMethodKey(data.method) as PaymentsKeyType);

        if (isPaymentPlatformPay(data)) {
            return {
                name,
                icon: data.method as PlatformPay,
            };
        }

        return { name };
    }, [data]);

    const renderLeftAddon = React.useMemo(() => {
        const { method } = data;

        if (method === 'cashback_edadeal') {
            return <Icon name="edadeal" width={24} height={24} />;
        }

        if (method === 'new_promocode' || method === 'virtual_kinopoisk') {
            return <Icon name="promocode" width={24} height={24} />;
        }

        if (method === 'yandex_account_withdraw' || method === 'yandex_account_topup') {
            return <Icon name="plusColorful" width={24} height={24} />;
        }

        return <Icon name="creditCard" width={24} height={24} />;
    }, [data]);

    const renderRightAddon = React.useMemo(() => {
        if (icon) {
            return <Icon name={icon} width={36} height={24} />;
        }

        const { price, method } = data;

        if (method === 'payment' || !isStrictDefined(price)) {
            return null;
        }

        if (method === 'yandex_account_withdraw') {
            return <PlusValue value={-price} mode="medium" refunded={refunded} />;
        }

        if (method === 'yandex_account_topup') {
            return <PlusValue value={price} mode="medium" refunded={refunded} />;
        }

        if (method === 'cashback_edadeal') {
            return <CashbackValue value={price} mode="medium" />;
        }

        return <div className={styles.payment__price}>{formatPrice(price, data.currency, refunded)}</div>;
    }, [icon, data, refunded]);

    return (
        <div className={styles.payment}>
            {renderLeftAddon}
            <div className={styles.payment__name}>{name}</div>
            {renderRightAddon}
        </div>
    );
};

export default Payment;
