import i18nFactory from '@i18n';
import * as keyset from '@i18n/ReceiptList.i18n';
import React, { useCallback, useMemo, useState } from 'react';

import { cn } from '@bem-react/classname';

import { formatPrice } from '@client/helpers/number';
import { useDateAutoFormatter } from '@hooks/useDateFormatters';
import useMetrika from '@hooks/useMetrika';
import { IReceipt } from '@src/types';

import './index.css';
import { ArrowIcon, ReceiptIcon } from './icons';

const cnReceiptList = cn('ReceiptList');

const SkeletonCn = cnReceiptList('SkeletonItem');

const i18n = i18nFactory(keyset);

interface IReceiptItem {
    amount?: string;
    date?: string;
    url: string;
    onClick(): void;
}

const Skeleton = () => (
    <div className={cnReceiptList('Skeleton')}>
        <div className={cnReceiptList('SkeletonIcon', [SkeletonCn])} />
        <div className={cnReceiptList('SkeletonDescription', [SkeletonCn])} />
    </div>
);

const ReceiptItem = React.memo<IReceiptItem>(({ amount, date, url, onClick }) => {
    const formatDate = useDateAutoFormatter();

    const datetime = React.useMemo(() => {
        if (!date) {
            return;
        }

        return formatDate(date, { timeFormat: ', HH:mm' });
    }, [date, formatDate]);

    return (
        <a className={cnReceiptList('Item')} href={url} target="_blank" rel="noopener noreferrer" onClick={onClick}>
            <div className={cnReceiptList('Icon')}>
                <ReceiptIcon className={cnReceiptList('ReceiptIcon')} />
            </div>
            <div className={cnReceiptList('Description')}>
                {
                    amount ? (
                        <>
                            <span className={cnReceiptList('Price')}>{amount}</span>
                            <span className={cnReceiptList('Date')}> – {datetime}</span>
                        </>
                    ) : (
                        <span className={cnReceiptList('Price')}>Открыть чек</span>
                    )
                }
            </div>
            <div className={cnReceiptList('Arrow')}>
                <ArrowIcon className={cnReceiptList('ArrowIcon')} />
            </div>
        </a>
    );
});

interface IReceiptProps {
    receipts?: IReceipt[];
    currency: string;
    metrikaParams?: string | object;
}

export const ReceiptList: React.FC<IReceiptProps> = ({ receipts, currency, metrikaParams }) => {
    const [showPrepayment, setShowPrepayment] = useState(false);
    const metrika = useMetrika();

    const onExpandReceipts = useCallback((e) => {
        e.preventDefault();
        setShowPrepayment(true);

        metrika.params({
            click_receipt_expand: metrikaParams,
        });
    }, [setShowPrepayment, metrikaParams, metrika]);

    const onItemClick = useCallback((type) => () => {
        metrika.params({
            [`click_receipt_${type}`]: metrikaParams,
        });
    }, [metrika, metrikaParams]);

    const filteredReceipts = useMemo(() =>
        receipts?.filter(({ type }) => showPrepayment || type !== 'prepayment'),
    [receipts, showPrepayment]) || [];

    return (
        <div className={cnReceiptList()}>
            {receipts ?
                filteredReceipts.map(({ amount, timestamp, url, type }) =>
                    <ReceiptItem
                        key={`${url}_${timestamp}`}
                        amount={amount?.map((item) => formatPrice(Number(item), currency, false, { signed: false })).join(' + ')}
                        date={timestamp}
                        url={url}
                        onClick={onItemClick(type)}
                        />) :
                <Skeleton />
            }
            {receipts && filteredReceipts.length !== receipts.length && !showPrepayment && (
                <a className={cnReceiptList('Item')} onClick={onExpandReceipts} href="#">
                    <div className={cnReceiptList('Icon')}>
                        <ReceiptIcon className={cnReceiptList('ReceiptIcon', { type: 'secondary' })} />
                    </div>
                    <span className={cnReceiptList('ExpandText')}>{i18n('Показать все документы')}</span>
                </a>
            )}
        </div>
    );
};
