import i18nFactory from '@i18n';
import * as keyset from '@i18n/MainScreen.i18n';
import React, { useCallback, useState } from 'react';
import type { ReactNode } from 'react';

import { cn } from '@bem-react/classname';

import type { OrderPayment } from '@client/helpers/order';
import { compareOrderPayments, methodSortingOrder } from '@client/helpers/order';
import { i18nCommon } from '@client/translations/common';
import Button from '@components/Button';
import ButtonLink from '@components/ButtonLink';
import DropDownList from '@components/DropDownList';
import { useDateAutoFormatter } from '@hooks/useDateFormatters';
import { IOrder } from '@src/types';

import Detail from './Detail';
import styles from './index.module.css';
import Payment from './Payment';
import ReceiptList from './ReceiptList';
import './index.css';

const i18n = i18nFactory(keyset);

interface Props {
    created: string;
    fake?: Nullable<boolean>;
    payments?: Nullable<OrderPayment[]>;
    details: IOrder['items'];
    refunds: IOrder['refunds'];
    checksEnabled?: Nullable<boolean>;
    serviceUrl?: Nullable<string>;
    helpUrl?: Nullable<string>;
    refunded: boolean;
    alias?: string;
    purchaseToken: string;
    currency: string;
    isPushView?: boolean;
    isReceipt?: boolean;
    receiptUrl?: string;
    status: IOrder['status'];
}

const cnOrderInfo = cn('OrderInfo');

const OrderInfo: React.FC<Props> = ({
    refunded,
    created,
    payments,
    details,
    refunds,
    checksEnabled,
    serviceUrl,
    helpUrl,
    fake,
    alias,
    purchaseToken,
    currency,
    isPushView,
    isReceipt,
    receiptUrl,
    status,
}) => {
    const formatDate = useDateAutoFormatter();
    const [receiptShow, setReceitShow] = useState(true);

    const datetime = React.useMemo(() => {
        return formatDate(created, { timeFormat: ', HH:mm' });
    }, [created, formatDate]);

    const hasRefundsItems = React.useMemo(() => {
        return Boolean(refunds && refunds.some((item) => item && item.items && item.items.length > 0));
    }, [refunds]);

    const paymentMethods = React.useMemo(() => {
        return payments && payments.length ? (
            <div className={styles.payments}>
                {[...payments]
                    .sort(compareOrderPayments('method', methodSortingOrder))
                    .map((payment, index) => (
                        <Payment data={payment} refunded={refunded} key={`${payment.method}_${index}`} />
                    ))
                }
            </div>
        ) : null;
    }, [payments, refunded]);

    const metrikaAlias = alias ? { [alias]: true } : true;

    const onReceiptEmpty = useCallback(() => {
        setReceitShow(false);
    }, [setReceitShow]);

    const detailsList = React.useMemo(() => {
        const all_details: ReactNode[] = [];

        details && details.forEach((detail, i) => {
            all_details.push(<Detail data={detail} created={created} key={detail.name + i} />);
        });

        refunds && refunds.forEach((refund, i) => {
            refund.items?.forEach((refundItem, k) => {
                all_details.push(<Detail data={refundItem} created={created} refunded key={`${refundItem.name}_${i}_${k}`} />);
            });
        });

        return all_details;
    }, [details, refunds, created]);

    const getStatusText = React.useMemo(() => {
        if (!status) {
            return null;
        }

        return ({
            advance: i18n('Авансовый чек — это формальное подтверждение покупки. Позже придёт финальный.'),
        } as Record<string, string>)[status];
    }, [status]);

    return (
        <div className={styles.container}>
            {fake ? null : <div className={styles.dateline}>{datetime}</div>}

            {paymentMethods}

            {(detailsList && detailsList.length) || hasRefundsItems || isPushView ? (
                <div className={styles.details}>
                    <div className={styles.headline}>{i18nCommon('payments_details')}</div>
                    {detailsList && detailsList.length ?
                        <DropDownList count={5} data={detailsList} /> :
                        <div className={cnOrderInfo('DetailsHint')}>{i18n('Информация появится в разделе Платежи Яндекс ID')}</div>
                    }
                </div>
            ) : null}

            {receiptShow && checksEnabled && !isPushView && !isReceipt && (
                <div className={styles.details}>
                    <div className={styles.headline}>Чеки</div>
                    <ReceiptList
                        purchaseToken={purchaseToken}
                        currency={currency}
                        onReceiptEmpty={onReceiptEmpty}
                        metrikaParams={{ service: metrikaAlias }}
                    />
                </div>
            )}

            {isReceipt && receiptUrl && (
                <div className={styles.details}>
                    <div className={styles.headline}>Чеки</div>
                    <ReceiptList
                        data={[{
                            url: receiptUrl,
                        }]}
                        currency={currency}
                        metrikaParams={{ service: metrikaAlias }}
                    />
                </div>
            )}

            {getStatusText && (
                <p className={styles.statusDescription}>
                    {getStatusText}
                </p>
            )}

            {(serviceUrl || helpUrl) && !isPushView ? (
                <div className={styles.buttons}>
                    {serviceUrl ? (
                        <ButtonLink
                            className={styles.button}
                            link={serviceUrl}
                            text={i18nCommon('to_service')}
                            icon="arrowRight"
                            metrika={{ service: metrikaAlias }}
                        />
                    ) : null}
                    {helpUrl ? (
                        <ButtonLink
                            className={styles.button}
                            link={helpUrl}
                            text={i18nCommon('support')}
                            icon="help"
                            metrika={{ service: metrikaAlias }}
                        />
                    ) : null}
                </div>
            ) : null}

            {isPushView && (
                <Button className={cnOrderInfo('CloseButton')} type="link" size="m" view="default" url="/am/finish?status=ok" width="max">
                    {i18nCommon('close')}
                </Button>
            )}
        </div>
    );
};

export default OrderInfo;
