import React, { useMemo } from 'react';

import * as orderHelpers from '@client/helpers/order';
import { i18nCommon } from '@client/translations/common';
import DetailLayout from '@components/DetailLayout';
import { useSelector } from '@hooks/redux';
import useConstants from '@hooks/useConstants';
import { usePassportAvatar } from '@hooks/usePassportAvatar';
import { useServiceName } from '@hooks/useServiceName';
import { IOrder } from '@src/types';

import OrderInfo from './OrderContent';
import OrderHeader from './OrderHeader';

import { isRefunded } from '../../../helpers';
import { isPlusTransaction } from '../../../helpers/order';
import { getDescription } from '../helper';

type Props = {
    data: IOrder;
};

const OrderDetails = ({ data }: Props) => {
    const { plusServiceData, familyUsers, lang } = useConstants();
    const isPushView = useSelector((state) => Boolean(state.orders.order));
    const { initiatorUid } = data;
    const initiator = initiatorUid && familyUsers?.[initiatorUid] ? familyUsers?.[initiatorUid] : undefined;
    const avatarUrl = usePassportAvatar(initiator?.avatar);

    const isPlusTopup = useMemo(() => orderHelpers.isPlusTopup(data), [data]);
    const items = useMemo(() => {
        if (!data) {
            return [];
        }

        const result = data.items?.filter(({ hidden }) => !hidden);

        // Не хотим показывать Доставку, Сервисный сбор и т.п. в деталях, если кроме них ничего нет
        if (result?.length === 1 && orderHelpers.isHiddenAlias(result[0].alias)) {
            return [];
        }

        return result;
    }, [data]);

    const serviceName = useServiceName(data.service.name, data.service.i18nKeyName);
    let customServiceName = '';

    if (isPlusTopup) {
        customServiceName = i18nCommon('plus_service_name');
    } else if (data.isReceipt) {
        customServiceName = data.retailerName || i18nCommon('my_purchase');
    } else if (data.source === 'yandexpay') {
        customServiceName = i18nCommon('yandexpayBuy');
    }

    const renderHeader = React.useCallback(() => {
        if (!data) {
            return null;
        }

        const isPlus = isPlusTransaction(data);

        const description = isPlusTopup ? serviceName : getDescription(data);

        return (
            <OrderHeader
                initiatorAvatar={avatarUrl}
                initiatorName={initiator?.name}
                iconUrl={isPlusTopup ? plusServiceData.iconUrl : (data.retailerIcon || data.service.iconUrl)}
                splashUrl={isPlusTopup ? plusServiceData.splashUrl : data.service.splashUrl}
                name={customServiceName || serviceName}
                details={description}
                price={isPlus ? undefined : data.total}
                status={data.status}
                currency={data.currency}
                plus={data.plus}
                isReceipt={data.isReceipt}
                lang={lang}
                isPushView={isPushView}
            />
        );
    }, [data, serviceName, isPlusTopup, plusServiceData, initiator, avatarUrl, lang, isPushView]);

    const renderInfo = React.useCallback(() => {
        if (!data) {
            return null;
        }
        const isOnlyPlusTrancsaction = isPlusTopup || (data.plus !== 0 && data.total === 0);

        return (
            <OrderInfo
                created={data.created}
                payments={data.payments}
                details={items}
                purchaseToken={data.purchaseToken}
                currency={data.currency}
                checksEnabled={!isOnlyPlusTrancsaction && data.checksEnabled}
                helpUrl={data.isReceipt ? undefined : data.service.help}
                serviceUrl={data.isReceipt ? undefined : data.service.url}
                refunds={data.status !== 'cancelled' ? data.refunds : []}
                fake={data.fake}
                refunded={isRefunded(data.status)}
                alias={data.service.alias}
                isPushView={isPushView}
                isReceipt={data.isReceipt}
                receiptUrl={data.receiptUrl}
                status={data.status}
            />
        );
    }, [data]);

    if (!data) {
        return null;
    }

    return <DetailLayout renderHeader={renderHeader} renderInfo={renderInfo} />;
};

export default React.memo(OrderDetails);
