import React from 'react';

import { cn } from '@bem-react/classname';

import { isRefunded, isShowStatus } from '@client/helpers';
import { formatPlus, formatPrice } from '@client/helpers/number';
import { i18nCommon } from '@client/translations/common';
import Button from '@components/Button';
import Icon from '@components/Icon';
import Image from '@components/Image';
import StatusOrder from '@components/StatusOrder';
import { isStrictDefined } from '@src/helpers';
import { Lang } from '@src/types';

import styles from './index.module.css';

import './index.css';

interface Props {
    splashUrl?: Nullable<string>;
    iconUrl?: Nullable<string>;
    name: string;
    details?: string;
    price?: Nullable<Client.Price>;
    status?: Nullable<Client.StatusOrder>;
    plus?: Nullable<number>;
    currency?: Nullable<string>;
    initiatorAvatar?: string;
    initiatorName?: string;
    lang?: Lang;
    isPushView?: boolean;
    isReceipt?: boolean;
}

const cnOrderHeader = cn('OrderHeader');

const HeaderPlus: React.FC<{ plus?: Nullable<number>; refunded: boolean; }> = ({ plus, refunded }) => {
    if (!isStrictDefined(plus)) {
        return null;
    }

    const isDeducting = plus < 0;

    const formattedPlus = [
        formatPlus(plus, refunded, { leadPlus: true, signed: true }),
        isDeducting && i18nCommon('points', { count: Math.abs(plus) }),
    ].filter(Boolean).join(' ');

    return (
        <div className={styles.price}>
            {formattedPlus}
        </div>
    );
};

const OrderHeader: React.FC<Props> = ({
    splashUrl, iconUrl, name, details, price, status, plus, currency, initiatorName, initiatorAvatar,
    isPushView, lang, isReceipt,
}) => {
    const refunded = isRefunded(status);

    const renderServiceLogo = React.useCallback((size: number, className?: string) => {
        return iconUrl ?
            <Image cover width={size} height={size} className={className} src={iconUrl} /> :
            <Icon cover width={size} height={size} className={className} name={isPushView || isReceipt ? 'shop' : 'Yandex'} />;
    }, [iconUrl, isPushView, isReceipt]);

    return (
        <div className={cnOrderHeader({ view: isPushView ? 'push' : undefined })}>
            {splashUrl ? <Image cover src={splashUrl} className={styles.substrate} /> : null}
            {isReceipt ? <div className={styles.receiptBackground} /> : null}
            <div className={cnOrderHeader('Order', { view: isPushView ? 'push' : undefined })}>
                {isPushView && (
                    <div className={cnOrderHeader('IdLogo', { lang })} />
                )}
                <div className={cnOrderHeader('LogoContainer')}>
                    {initiatorAvatar ? (
                        <>
                            <Image cover width={44} height={44} className={cnOrderHeader('Logo')} src={initiatorAvatar} />
                            {renderServiceLogo(16, cnOrderHeader('Logo', { additional: true }))}
                        </>
                    ) : renderServiceLogo(44, cnOrderHeader('Logo'))}
                </div>
                {(details || initiatorName || name) &&
                    <div className={styles.info}>
                        <div className={styles.name}>{[initiatorName, name].filter(Boolean).join(' • ')}</div>
                        {details && <div className={styles.details}>{details}</div>}
                    </div>
                }
                {isStrictDefined(price) && price !== 0 ? (
                    <div className={styles.price}>{formatPrice(price, currency, refunded)}</div>
                ) : (
                    <HeaderPlus plus={plus} refunded={refunded} />
                )}

                {isShowStatus(status) ? (
                    <div className={styles.status}>
                        <StatusOrder
                            extended
                            size="s16"
                            status={status}
                            className={styles.status__block}
                            classNameText={styles.status__text}
                        />
                    </div>
                ) : null}

                {isPushView && (
                    <Button
                        className={cnOrderHeader('Close')}
                        type="link"
                        size="l"
                        view="clear"
                        url="/am/finish?status=ok"
                        icon={(className) => <div className={cnOrderHeader('CloseIcon', [className])} />} />
                )}
            </div>
        </div>
    );
};

export default OrderHeader;
