import React, { memo, useCallback, useMemo } from 'react';
import { Link } from 'react-router-dom';

import * as orderHelpers from '@client/helpers/order';
import { i18nCommon } from '@client/translations/common';
import Card from '@components/Card';
import useConstants from '@hooks/useConstants';
import useMetrika from '@hooks/useMetrika';
import { usePassportAvatar } from '@hooks/usePassportAvatar';
import useRoutes from '@hooks/useRoutes';
import { useServiceName } from '@hooks/useServiceName';
import { IOrder } from '@src/types';

import styles from './index.module.css';

import { getDescription, getDetailName } from '../../helper';

type Props = {
    data: IOrder;
};

const MainScreenCard: React.FC<Props> = ({ data }) => {
    const routes = useRoutes();
    const metrika = useMetrika();
    const { plusServiceData, familyUsers } = useConstants();
    const isPlus = useMemo(() => {
        return orderHelpers.isPlusTransaction(data);
    }, [data]);

    const isPlusTopup = useMemo(() => orderHelpers.isPlusTopup(data), [data]);

    const description = useMemo(() => {
        if (!data.items) {
            return '';
        }

        const customDescription = getDescription(data);

        if (customDescription) {
            return customDescription;
        }

        const goods = data.items
            .slice()
            .sort((a, b) => b.price - a.price)
            .filter(({ alias }) => !orderHelpers.isHiddenAlias(alias));
        const goodsLength = goods?.length;

        let description = goods?.filter(({ hidden }) => !hidden).map((item) => getDetailName(item, data.created)).join(', ');

        if (goodsLength && goodsLength > 1) {
            const i18nKey = data.service.i18nKeyName === 'afisha' ? 'tikets' : 'goods';

            description = [
                `${goodsLength} ${i18nCommon(i18nKey, { count: goodsLength })}`,
                description,
            ].filter(Boolean).join(': ');
        }

        return description;
    }, [data]);

    const serviceName = useServiceName(data.service.name, data.service.i18nKeyName);

    let customServiceName = '';

    if (isPlusTopup) {
        customServiceName = i18nCommon('plus_service_name');
    } else if (data.isReceipt) {
        customServiceName = data.retailerName || i18nCommon('my_purchase');
    } else if (data.source === 'yandexpay') {
        customServiceName = i18nCommon('yandexpayBuy');
    }

    const to = useCallback(
        (currentLocation: Location) => {
            return routes.order(currentLocation, data);
        },
        [data, routes],
    );

    const { orderId, initiatorUid } = data;

    const handleClick = useCallback(() => {
        const paymentMethods = data.payments?.reduce<Record<string, object>>((aux, payment) => {
            aux[payment.method] = {
                currency: payment.currency,
                amount: payment.price,
            };

            return aux;
        }, {});

        metrika.params({
            click_order: {
                [orderId]: {
                    service_alias: data.isReceipt ? 'receipts' : data.service.alias,
                    status: data.status,
                    payment_methods: paymentMethods,
                },
            },
        });
    }, [metrika, orderId, data]);

    const initiator = initiatorUid && familyUsers?.[initiatorUid] ? familyUsers?.[initiatorUid] : undefined;
    const avatarUrl = usePassportAvatar(initiator?.avatar);

    return (
        <Link to={to} className={styles.card} onClick={handleClick}>
            <Card
                initiatorAvatar={avatarUrl}
                initiatorName={initiator?.name}
                serviceName={customServiceName || serviceName}
                icon={isPlusTopup ? plusServiceData.iconUrl : (data.retailerIcon || data.service.iconUrl)}
                description={isPlusTopup ? serviceName : description}
                price={isPlus ? undefined : data.total}
                status={data.status}
                plus={data.plus}
                currency={data.currency}
                isReceipt={data.isReceipt}
                cashback={data.cashback}
            />
        </Link>
    );
};

export default memo(MainScreenCard);
