import i18nFactory from '@i18n';
import * as keyset from '@i18n/MainScreen.i18n';
import qs from 'query-string';
import React, { memo, useCallback, useEffect } from 'react';
import { useDispatch } from 'react-redux';
import { useHistory, useLocation } from 'react-router';

import { Text } from '@yandex-int/tap-components/Skeleton';

import OrdersListHeaderLayout from '@client/components/OrdersListHeaderLayout';
import { setFamilyFilter } from '@client/redux/orders';
import { Tabs } from '@components/Tabs';
import { useSelector } from '@hooks/redux';
import useConstants from '@hooks/useConstants';
import useRoutes from '@hooks/useRoutes';

import FilterList from './FilterList';
import FilterListSkeleton from './FilterList/Skeleton';

import { getQueryParams } from '../List/helpers';

const i18n = i18nFactory(keyset);

const MainScreenComponentTop: React.FC = () => {
    const dispatch = useDispatch();
    const { search, pathname } = useLocation();
    const { tld, flags: { fnsEnabled }, isLite, familyMode } = useConstants();
    const history = useHistory();
    const queryParams = getQueryParams(search);
    const routes = useRoutes();
    const alias = useSelector((state) => state.orders.alias);
    const familyFilter = useSelector((state) => state.orders.familyFilter);

    useEffect(() => {
        const query = qs.parse(history.location.search);

        query.alias = alias !== 'receipts' ? alias : undefined;
        query.familyFilter = familyFilter;
        isLite && (query.isLite = '1');
        familyMode && (query.family = '1');

        history.replace({
            search: qs.stringify(query),
        });
    }, [alias, familyFilter, familyMode, isLite, history]);

    const services = useSelector((state) => state.orders.services);

    const onChangeTab = useCallback<React.ChangeEventHandler<HTMLInputElement>>((event) => {
        history.replace(event.target.value);
        dispatch({ type: 'CHANGE_ALIAS', payload: alias !== 'receipts' ? 'receipts' : null });
        dispatch(setFamilyFilter());
    }, [history, dispatch, alias]);

    if (!services) {
        return (
            <OrdersListHeaderLayout title={<Text width="40%" size={24} />} banner={<Text width="70%" size={20} />}>
                <FilterListSkeleton />
            </OrdersListHeaderLayout>
        );
    }

    return (
        <>
            <OrdersListHeaderLayout
                title={i18n('История платежей')}
                banner={
                    tld === 'kz' || tld === 'by' ?
                        i18n('Сейчас здесь только баллы Плюса. Остальные платежи появятся позже.') :
                        i18n('Некоторые платежи могут быть не видны. Но мы обязательно их добавим.')
                }
            >
                {fnsEnabled && <Tabs value={history.location.pathname} onChange={onChangeTab} />}
                {pathname === routes.main() && (
                    <FilterList data={services} active={queryParams.alias} />
                )}
            </OrdersListHeaderLayout>
        </>
    );
};

export default memo(MainScreenComponentTop);
