import React, { memo, useCallback, useEffect } from 'react';

import { i18nCommon } from '@client/translations/common';
import CardSceleton from '@components/Card/Skeleton';
import { EdadealLanding } from '@components/EdadealLanding';
import ListDivider from '@components/ListDivider';
import ListOrders from '@components/ListOrders';
import ListOrdersItem from '@components/ListOrders/ListOrdersItem';
import { Stub } from '@components/Stub';
import { useDispatch, useSelector } from '@hooks/redux';
import useConstants from '@hooks/useConstants';
import useMetrika from '@hooks/useMetrika';
import { useRequest } from '@hooks/useRequest';
import { useErrors } from '@lib/errors';
import { IOrder } from '@src/types';

import { getStubProps } from './helpers';
import styles from './index.module.css';
import OrdersListSkeleton from './Skeleton';

import Anchor from '../Anchor';
import Card from '../Card';

function renderCard(node: IOrder) {
    return <Card data={node} />;
}

function getISODate(node: IOrder) {
    return node.created;
}

const OrdersListContainer = () => {
    const request = useRequest();
    const dispatch = useDispatch();
    const { fnsData, familyMode, familyUsers, authInfo } = useConstants();
    const alias = useSelector((state) => state.orders.alias);
    const isUpdating = useSelector((state) => state.orders.isUpdating);
    const isUpdateFailed = useSelector((state) => state.orders.isUpdateFailed);
    const edges = useSelector((state) => state.orders.orders?.edges);
    const endCursor = useSelector((state) => state.orders.orders?.pageInfo?.endCursor);
    const hasNextPage = useSelector((state) => state.orders.orders?.pageInfo?.hasNextPage);
    const services = useSelector((state) => state.orders.services);
    const firstLoading = useSelector((state) => state.orders.firstLoading);
    const familyFilter = useSelector((state) => state.orders.familyFilter);
    const { addError } = useErrors();
    const metrika = useMetrika();

    const renderUpdateNotification = useCallback(() => {
        return (
            <>
                <ListOrdersItem>
                    <ListDivider text={i18nCommon('loading_list')} />
                </ListOrdersItem>
                <ListOrdersItem>
                    <CardSceleton />
                </ListOrdersItem>
            </>
        );
    }, []);

    const onFetchMore = useCallback(() => {
        dispatch({ type: 'UPDATE_ORDERS_START' });

        const params: Record<string, string | undefined | null> = { alias, after: endCursor };

        if (familyFilter) {
            params.initiatorUids = familyFilter;
        } else if (familyMode && familyUsers) {
            params.initiatorUids = Object.keys(familyUsers).join(',') || authInfo?.user.uid;
        }

        request.get('/orders', { params }).then((response) => {
            metrika.params({
                page_loaded: true,
            });
            dispatch({ type: 'FETCH_MORE', payload: response.data });
        }).catch((error) => {
            addError({
                refetch: onFetchMore,
                error,
            });
            dispatch({ type: 'FETCH_FAILED' });
        });
    }, [
        endCursor,
        alias,
        dispatch,
        request,
        addError,
        metrika,
        familyUsers,
        familyMode,
        familyFilter,
        authInfo?.user.uid,
    ]);

    useEffect(() => {
        if (edges) {
            return;
        }

        const params: Record<string, string | undefined> = { alias };

        if (familyFilter) {
            params.initiatorUids = familyFilter;
        } else if (familyMode && familyUsers) {
            params.initiatorUids = Object.keys(familyUsers).join(',') || authInfo?.user.uid;
        }

        dispatch({ type: 'UPDATE_ORDERS_START' });
        request.get('/orders', { params })
            .then((response) => {
                dispatch({ type: 'UPDATE_ORDERS', payload: response.data });
            }).catch((error) => {
                addError({
                    refetch: () => {
                        window.location.reload();
                    },
                    error,
                });
                dispatch({ type: 'FETCH_FAILED' });
            });
    }, [edges, alias, request, dispatch, addError, familyMode, familyUsers, familyFilter, authInfo?.user.uid]);

    useEffect(() => {
        if (firstLoading) {
            return;
        }

        const isEmpty = !edges || edges.length === 0;
        const goalPrefix = `show_order-list_${alias || 'all'}`;

        metrika.params({
            'order-list-show': true,
            [isEmpty ? goalPrefix + '_empty' : goalPrefix]: true,
        });
    }, [alias, firstLoading]);

    if (!edges) {
        return <OrdersListSkeleton />;
    }

    if (edges.length === 0) {
        if (alias === 'receipts' && !fnsData?.isBind) {
            return <EdadealLanding />;
        }

        return <Stub className={styles.stub} alias={alias} {...getStubProps(alias, services)} />;
    }

    return (
        <>
            <ListOrders
                list={edges}
                isUpdating={isUpdating}
                renderRow={renderCard}
                renderUpdateNotification={renderUpdateNotification}
                getISODate={getISODate}
            />
            {!isUpdating && !isUpdateFailed && hasNextPage ? <Anchor onHit={onFetchMore} /> : null}
        </>
    );
};

export default memo(OrdersListContainer);
