import i18nFactory from '@i18n';
import * as keyset from '@i18n/MainScreen.i18n';

import type { OrderDetail } from '@client/helpers/order';
import { i18nCommon } from '@client/translations/common';
import { IOrder } from '@src/types';

type HourOfDay = 'night' | 'morning' | 'afternoon' | 'evening';

const i18n = i18nFactory(keyset);

const hourOfDay = (time: number): HourOfDay => {
    if (time >= 23 || time < 5) {
        return 'night';
    }

    if (time >= 5 && time < 12) {
        return 'morning';
    }

    if (time >= 12 && time < 18) {
        return 'afternoon';
    }

    return 'evening';
};

const detailAliases = {
    delivery: () => i18nCommon('delivery'),
    tip: () => i18nCommon('gratuity'),
    topup: () => i18nCommon('adding_account'),
    fine: () => i18n('Штраф, возмещение убытков'),
    'toll-road': () => i18n('Проезд по платной дороге'),
    trip: (created: string) => {
        const period = hourOfDay(new Date(created).getHours());

        return {
            night: () => i18n('Поездка ночью'),
            morning: () => i18n('Поездка утром'),
            afternoon: () => i18n('Поездка днём'),
            evening: () => i18n('Поездка вечером'),
        }[period]();
    },
    ride: (created: string) => {
        const period = hourOfDay(new Date(created).getHours());

        return {
            night: () => i18n('Ночной заказ'),
            morning: () => i18n('Утренний заказ'),
            afternoon: () => i18n('Дневной заказ'),
            evening: () => i18n('Вечерний заказ'),
        }[period]();
    },
};

type DetailAliases = keyof typeof detailAliases;

const getDetailByAlias = (alias: string | undefined | null, created: string): string => {
    const aliasExist = alias ? alias in detailAliases : false;

    return aliasExist ? detailAliases[alias as DetailAliases](created) : '';
};

const getDriveDescription = (data: IOrder): string => {
    const productName = data.items && data.items.length > 0 ? data.items[0].name : undefined;
    const alias = data.items && data.items.length > 0 ? data.items[0].alias : undefined;

    if (!productName) {
        return '';
    }

    return getDetailByAlias(alias, data.created) || productName;
};

const getGoDescription = (data: IOrder) => {
    const product = data.items ? data.items[0] : undefined;
    const alias = data.items && data.items.length > 0 ? data.items[0].alias : undefined;

    if (!product) {
        return '';
    }

    const period = hourOfDay(new Date(data.created).getHours());

    if (product.name === 'Услуги курьерской доставки') {
        return {
            night: () => i18n('Доставка ночью'),
            morning: () => i18n('Доставка утром'),
            afternoon: () => i18n('Доставка днём'),
            evening: () => i18n('Доставка вечером'),
        }[period]();
    }

    return getDetailByAlias(alias, data.created);
};

export const getDescription = (data: IOrder) => {
    const { service: { i18nKeyName } } = data;

    switch (i18nKeyName) {
        case 'drive':
            return getDriveDescription(data);

        case 'taxi':
            return getGoDescription(data);

        case 'help_nearby':
            return i18n('Спасибо, что помогаете!');

        default:
            return '';
    }
};

export const getDetailName = (data: OrderDetail, created: string): string => {
    return getDetailByAlias(data.alias, created) || data.name;
};
