/* eslint-disable no-console */

const isLocalhost = window.location.hostname.startsWith('localhost');

type Config = {
    scope?: string;
    onUpdate?: (registration: ServiceWorkerRegistration) => void;
    onSuccess?: (registration: ServiceWorkerRegistration) => void;
};

const { baseUrl } = window.__CONSTANTS__;

const registerValidSW = (swUrl: string, config?: Config) => {
    navigator.serviceWorker
        .register(swUrl, { scope: config?.scope })
        .then((registration) => {
            registration.onupdatefound = () => {
                const installingWorker = registration.installing;

                if (!installingWorker) {
                    return;
                }
                installingWorker.onstatechange = () => {
                    if (installingWorker.state === 'installed') {
                        if (navigator.serviceWorker.controller) {
                            console.log(
                                'New content is available and will be used when all tabs for this page are closed',
                            );

                            config?.onUpdate?.(registration);
                        } else {
                            console.log('Content is cached for offline use.');

                            config?.onSuccess?.(registration);
                        }
                    }
                };
            };
        })
        .catch((error) => {
            console.error('Error during service worker registration:', error);
        });
};

const checkValidServiceWorker = (swUrl: string, config?: Config) => {
    fetch(swUrl, {
        headers: { 'Service-Worker': 'script' },
    })
        .then((response) => {
            response.status === 404 || (response.headers.get('content-type') || '').indexOf('javascript') === -1 ?
                navigator.serviceWorker.ready.then((registration) => {
                    registration.unregister().then(() => {
                        window.location.reload();
                    });
                }) :
                registerValidSW(swUrl, config);
        })
        .catch(() => {
            console.log('No internet connection found. App is running in offline mode.');
        });
};

export const register = (config?: Config) => {
    if ('serviceWorker' in navigator && new URL(baseUrl, window.location.href).origin === window.location.origin) {
        window.addEventListener('load', () =>
            (isLocalhost ? checkValidServiceWorker : registerValidSW)(`${baseUrl}/sw.js`, config),
        );
    }
};

export function unregister() {
    if ('serviceWorker' in navigator) {
        navigator.serviceWorker.ready
            .then((registration) => {
                registration.unregister();
            })
            .catch((error) => {
                console.error(error.message);
            });
    }
}
