/// <reference lib="webworker" />
/* eslint-disable no-console */
type Manifest = {
    url: string;
    revision?: string | null;
    integrity?: string | null;
};

// eslint-disable-next-line @typescript-eslint/no-unused-vars
interface Window extends ServiceWorkerGlobalScope {
    __WB_MANIFEST: Manifest[];
    addEventListener(
        type: 'install' | 'activate',
        listener: (event: ExtendableEvent) => unknown,
        options?: boolean | AddEventListenerOptions,
    ): void;
    addEventListener(
        type: 'fetch',
        listener: (event: FetchEvent) => unknown,
        options?: boolean | AddEventListenerOptions,
    ): void;
}

const updateCacheParam = '__UPDATE__';
const precacheConfig = self.__WB_MANIFEST;
const cacheNamePrefix = 'sw-precache-v1-order-history-' + (self.registration?.scope || '') + '-';

const ignoreUrlParametersMatching = [/^utm_/];

const addDirectoryIndex = (originalUrl: string) => {
    const url = new URL(originalUrl);

    if (url.pathname.slice(-1) === '/') {
        url.pathname += 'index.html';
    }

    return url.toString();
};

const getCacheBustedUrl = (url: string, param: string | number) => {
    const urlWithCacheBusting = new URL(url);

    urlWithCacheBusting.search +=
        (urlWithCacheBusting.search ? '&' : '') + 'sw-precache=' + (param === updateCacheParam ? Date.now() : param);

    return urlWithCacheBusting.toString();
};

const populateCurrentCacheNames = (precacheConfig: Manifest[], cacheNamePrefix: string, baseUrl: string) =>
    precacheConfig.reduce<{
        absoluteUrlToCacheName: Record<string, string>;
        currentCacheNamesToAbsoluteUrl: Record<string, string>;
    }>(
        (acc, { url, revision }) => {
            const absoluteUrl = new URL(url, baseUrl).toString();
            const cacheName = cacheNamePrefix + absoluteUrl + '-' + (revision || '');

            acc.currentCacheNamesToAbsoluteUrl[cacheName] = absoluteUrl;
            acc.absoluteUrlToCacheName[absoluteUrl] = cacheName;

            return acc;
        },
        {
            absoluteUrlToCacheName: {},
            currentCacheNamesToAbsoluteUrl: {},
        },
    );

const stripIgnoredUrlParameters = (originalUrl: string, ignoreUrlParametersMatching: RegExp[]) => {
    const url = new URL(originalUrl);

    url.search = url.search
        .slice(1)
        .split('&')
        .map((kv) => kv.split('='))
        .filter((kv) => ignoreUrlParametersMatching.every((ignoredRegex) => !ignoredRegex.test(kv[0])))
        .map((kv) => kv.join('='))
        .join('&');

    return url.toString().trim();
};

const { absoluteUrlToCacheName, currentCacheNamesToAbsoluteUrl } = populateCurrentCacheNames(
    precacheConfig,
    cacheNamePrefix,
    self.location.origin,
);

const deleteAllCaches = () =>
    caches.keys().then((cacheNames) => Promise.all(cacheNames.map((cacheName) => caches.delete(cacheName))));

self.addEventListener('install', (event) => {
    event.waitUntil(
        Promise.all(
            Object.keys(currentCacheNamesToAbsoluteUrl).map((cacheName) =>
                caches.open(cacheName).then((cache) =>
                    cache.keys().then((keys) => {
                        const cacheBustParam = cacheName.split('-').pop();

                        if (keys.length && cacheBustParam !== updateCacheParam) {
                            return;
                        }

                        const absoluteUrl = currentCacheNamesToAbsoluteUrl[cacheName];
                        const urlWithCacheBusting = cacheBustParam ?
                            getCacheBustedUrl(absoluteUrl, cacheBustParam) :
                            absoluteUrl;
                        const request = new Request(urlWithCacheBusting, {
                            credentials: 'same-origin',
                            ...(absoluteUrl.indexOf('index.html') >= 0 ? { redirect: 'manual' } : {}),
                        });

                        return fetch(request, {
                            headers: {
                                'X-Requested-By': 'sw',
                            },
                        }).then((response) => {
                            if (response.ok) {
                                return cache.put(currentCacheNamesToAbsoluteUrl[cacheName], response);
                            }

                            console.error(
                                'Request for %s returned a response status %d, so not attempting to cache it.',
                                urlWithCacheBusting,
                                response.status,
                            );

                            caches.delete(cacheName);
                        });
                    }),
                ),
            ),
        )
            .then(() =>
                caches.keys().then((allCacheNames) =>
                    Promise.all(
                        allCacheNames.reduce<Promise<boolean>[]>((acc, cacheName) => {
                            if (!cacheName.indexOf(cacheNamePrefix) && !(cacheName in currentCacheNamesToAbsoluteUrl)) {
                                acc.push(caches.delete(cacheName));
                            }

                            return acc;
                        }, []),
                    ),
                ),
            )
            .then(() => void (typeof self.skipWaiting === 'function' && self.skipWaiting())),
    );
});

if (self.clients && typeof self.clients.claim === 'function') {
    self.addEventListener('activate', (event) => {
        event.waitUntil(self.clients.claim());
    });
}

self.addEventListener('message', (event) => {
    if (event.data.command === 'delete_all') {
        console.log('About to delete all caches...');
        deleteAllCaches()
            .then(() => {
                console.log('Caches deleted.');
                event.ports[0].postMessage({
                    error: null,
                });
            })
            .catch((error) => {
                console.log('Caches not deleted:', error);
                event.ports[0].postMessage({
                    error: error,
                });
            });
    }
});

self.addEventListener('fetch', (event) => {
    if (event.request.method !== 'GET') {
        return;
    }

    let urlWithoutIgnoredParameters = stripIgnoredUrlParameters(event.request.url, ignoreUrlParametersMatching);
    let cacheName = absoluteUrlToCacheName[urlWithoutIgnoredParameters];

    if (!cacheName) {
        urlWithoutIgnoredParameters = addDirectoryIndex(urlWithoutIgnoredParameters);
        cacheName = absoluteUrlToCacheName[urlWithoutIgnoredParameters];
    }

    if (cacheName) {
        event.respondWith(
            caches
                .open(cacheName)
                .then((cache) =>
                    cache.keys().then(([key]) => {
                        const getFromCache = () =>
                            cache.match(key).then((response) => {
                                if (response) {
                                    return response;
                                }

                                throw Error('The cache ' + cacheName + ' is empty.');
                            });

                        return cacheName.indexOf('index.html') >= 0 ?
                            fetch(event.request).catch(getFromCache) :
                            getFromCache();
                    }),
                )
                .catch((e) => {
                    console.warn('Couldn\'t serve response for "%s" from cache: %O', event.request.url, e);

                    return fetch(event.request);
                }),
        );
    }
});
