import * as path from 'path';

import * as csp from 'express-csp-header';

import * as secrets from './secrets.json';
import type { Config, CSPDirectives } from './types';

const baseUrl = '/order-history';
// eslint-disable-next-line @typescript-eslint/no-var-requires
const version = require('../../../.config/archon/lib/get-version');
const yastatic = 'https://yastatic.net/s3/passport-static/order-history';

const edadealHosts = [
    'https://desktop.checks.edadeal.yandex.ru',
    'https://desktop.checks.edastage.yandex.ru',
];

const isS3static = process.env.S3_STATIC === '1';
const clientBuild = path.resolve(__dirname, '../../../public/assets');

const common: Pick<
    Config,
    | 'bunker'
    | 'csrf'
    | 'version'
    | 'yandex'
    | 'blackbox'
    | 'trust'
    | 'passport'
    | 'edadeal'
    | 'images'
    | 'baseUrl'
    | 'paths'
    | 'local'
    | 'uatraits'
    | 'hs'
    | 'bills'
    | 'search'
> & {
    csp: CSPDirectives &
        Required<Pick<CSPDirectives, 'img-src' | 'connect-src' | 'frame-src' | 'script-src'>>;
    backend: Pick<Config['backend'], 'timeout'>;
    payments: Pick<Config['payments'], 'timeout'>;
    billsBackend: Pick<Config['billsBackend'], 'timeout'>;
} = {
    baseUrl,
    version,
    local: false,
    csrf: {
        salt: secrets.csrf_salt,
    },
    hs: {
        timeout: 4500,
        url: 'https://hs.yandex.ru/api',
    },
    yandex: {
        host: 'yandex.{tld}',
    },
    passport: {
        host: 'passport-test.yandex.{tld}',
    },
    edadeal: {
        hosts: edadealHosts,
    },
    images: {
        avatarHost: '//avatars.mds.yandex.net',
    },
    blackbox: {
        url: 'https://blackbox-test.yandex.net',
        timeout: 2500,
    },
    trust: {
        url: 'https://api.trust.test.yandex.net',
        timeout: 2500,
    },
    search: {
        url: 'https://msearch-proxy-test.search.yandex.net:10431',
        timeout: 4500,
    },
    payments: {
        timeout: 4500,
    },
    backend: {
        timeout: 4500,
    },
    billsBackend: {
        timeout: 20000,
    },
    uatraits: {
        type: 'http',
    },
    paths: {
        assets: isS3static ? yastatic : baseUrl + '/assets',
        versionized: `${yastatic}/v${version}`,
        public: path.resolve(__dirname, '../../../public'),
        clientBuild: clientBuild,
        serverBuild: path.resolve(__dirname, '../../../dist/root'),
        loadableStats: isS3static
            ? path.resolve(__dirname, '../../loadable-stats.json')
            : path.resolve(clientBuild, 'loadable-stats.json'),
    },
    bunker: {
        url: 'http://bunker-api-dot.yandex.net/v1',
        project: '/passport-order-history-test',
    },
    csp: {
        'default-src': [csp.NONE],
        'manifest-src': [csp.SELF],
        'style-src': [csp.SELF, 'yastatic.net', "'unsafe-inline'"],
        'font-src': [csp.SELF, 'yastatic.net'],
        'img-src': [
            csp.SELF,
            csp.DATA,
            'mc.admetrica.ru',
            'mc.yandex.ru',
            'avatars.mds.yandex.net',
            'yastatic.net',
            'leonardo.edadeal.io',
        ],
        'child-src': [csp.SELF],
        'frame-src': [csp.SELF, 'mc.yandex.md', 'mc.yandex.ru', ...edadealHosts],
        'script-src': [
            csp.SELF,
            csp.EVAL,
            csp.NONCE,
            'chat.s3.yandex.net',
            'mc.yandex.md',
            'mc.yandex.ru',
            'yastatic.net',
        ],
        'connect-src': [
            csp.SELF,
            'mc.yandex.ru',
            'https://yandex.ru/clck/click',
            /*
                Под капотом csp.TLD это строка %tld%, которую либа express-csp-header
                преобразует в нужное значение при каждом запросе
                @see https://github.com/frux/express-csp-header#auto-tld
            */
            `https://mail.yandex.${csp.TLD}`,
            `https://api.passport.yandex.${csp.TLD}`,
        ],
    },
    bills: {
        maxSelected: 100,
    },
};

export default common;
