import axios, { AxiosRequestConfig } from 'axios';
import { omit } from 'lodash';

import logger from './logger';

const normalizeParams: AxiosRequestConfig['paramsSerializer'] = (params = {}) => {
    return Object.entries(params)
        .reduce<string[]>((aux, entry) => {
            const [key, value] = entry;

            if (value !== null && value !== undefined) {
                if (Array.isArray(value)) {
                    value.forEach((item) => {
                        aux.push(`${key}=${encodeURIComponent(String(item))}`);
                    });

                    return aux;
                }

                aux.push(`${key}=${encodeURIComponent(String(value))}`);
            }

            return aux;
        }, [])
        .join('&');
};

export const createApiRequest = ({ baseURL, timeout, headers }: AxiosRequestConfig) => {
    const instance = axios.create({
        baseURL,
        timeout,
        headers,
        paramsSerializer: normalizeParams,
    });

    instance.interceptors.request.use(function (config) {
        const { baseURL, url, params } = config;
        const normalizedParams = normalizeParams(params);
        const requestedUrl = `${baseURL}${url}${normalizedParams ? '?' + normalizedParams : ''}`;

        logger.info({ config: omit(config, ['headers', 'data']) }, `Request → ${config.method?.toUpperCase()} : ${requestedUrl}`);

        return config;
    }, function (error) {
        if (error?.config?.headers) {
            delete error.config.headers;
        }

        logger.error({ error }, 'RequestError');

        return Promise.reject(error);
    });

    instance.interceptors.response.use(function (response) {
        const { status, statusText, config: { method, baseURL, url, params } } = response || {};
        const normalizedParams = normalizeParams(params);
        const requestedUrl = `${baseURL}${url}${normalizedParams ? '?' + normalizedParams : ''}`;

        logger.info(`${status} ${statusText} ← ${method?.toUpperCase()} ${requestedUrl}`);

        return response;
    }, function (error) {
        if (error?.config?.headers) {
            delete error.config.headers;
        }

        try {
            const { message, config: { method, baseURL, url, params } } = error;
            const normalizedParams = normalizeParams(params);
            const requestedUrl = `${baseURL}${url}${normalizedParams ? '?' + normalizedParams : ''}`;

            logger.error({ error }, `${message} ← ${method?.toUpperCase()} ${requestedUrl}`);
        } catch {
            logger.error({ error });
        }

        return Promise.reject(error);
    });

    return instance;
};
