import { AxiosRequestConfig, AxiosResponse } from 'axios';
import { Request, Response, RequestHandler } from 'express';

import { billReducer } from './reducers/bills';
import { documentReducer } from './reducers/documents';

import { ApiMiddleware } from '..';
import { ClientMpiInfo, IBillPayApiRequest, IBillsState, IDocumentsState } from '../../../../types';
import config from '../../../config';
import { createApiRequest } from '../../../lib/request';
import { apiErrorHandler } from '../../error/apiErrorHandler';

class BillsApi extends ApiMiddleware {
    constructor() {
        super();

        this.request = createApiRequest({
            baseURL: config.billsBackend.url,
            timeout: config.billsBackend.timeout,
            responseType: 'json',
        });
    }

    getApiPrefix = (): string => '/api/v1';

    getRequestOptions = (req: Request): AxiosRequestConfig => {
        const { tvm, blackbox } = req;
        const ticket = tvm?.frontend.tickets?.bills_backend?.ticket;
        const headers: AxiosRequestConfig['headers'] = {};

        if (req.id) {
            headers['X-Request-ID'] = req.id;
        }

        if (ticket) {
            headers['X-Ya-Service-Ticket'] = ticket;
        }

        if (blackbox?.userTicket) {
            headers['X-Ya-User-Ticket'] = blackbox.userTicket;
        }

        return { headers };
    };

    fullfillMpiInfo = (req: Request, clientInfo: ClientMpiInfo): BillsBackend.Mpi3dsInfo => {
        const { headers } = req;

        const ip = req.ip ?? headers['x-forwarded-for'];
        const browser_ip = Array.isArray(ip) ? ip[0] : ip.split(',')[0];
        const browser_accept_header = String(headers.accept);
        const browser_user_agent = String(headers['user-agent']);
        const browser_language = String(headers['accept-language']).split(',')[0];

        return {
            ...clientInfo,
            browser_ip,
            browser_accept_header,
            browser_user_agent,
            browser_language,
            browser_javascript_enabled: true,
        };
    };

    getDocumentsStateMiddleware = (req: Request, _res: Response): Promise<IDocumentsState> => {
        if (req.appContext.payServicesEnabled) {
            return this.request
                .get(`${this.getApiPrefix()}/documents`, this.getRequestOptions(req))
                .then((response: AxiosResponse<BillsBackend.DocumentsResponse>) => {
                    const documents = response.data.data.documents.map(documentReducer);

                    return {
                        state: 'stable',
                        list: documents,
                    };
                });
        }

        return Promise.resolve({
            state: 'initial',
            list: [],
        });
    };

    getBillsStateMiddleware = (req: Request, _res: Response): Promise<IBillsState> => {
        if (req.appContext.payServicesEnabled) {
            return this.request
                .get(`${this.getApiPrefix()}/bills`, this.getRequestOptions(req))
                .then((response: AxiosResponse<BillsBackend.BillsResponse>) => {
                    const state = response.data.data.state;
                    const bills = response.data.data.bills;

                    return {
                        state: state === 'completed' ? 'emulation' : state,
                        list: bills,
                        selected: [],
                    };
                });
        }

        return Promise.resolve({
            state: 'initial',
            list: [],
            selected: [],
        });
    };

    getDocuments: RequestHandler = (req, res, next) => {
        return this.request
            .get(`${this.getApiPrefix()}/documents`, this.getRequestOptions(req))
            .then((response: AxiosResponse<BillsBackend.DocumentsResponse>) => {
                const documents = response.data.data.documents.map(documentReducer);

                return res.json(documents);
            })
            .catch(apiErrorHandler(req, res, next));
    };

    createDocument: RequestHandler = (req, res, _next) => {
        const { type, value, title } = req.body;

        return this.request
            .post(
                `${this.getApiPrefix()}/documents`,
                { type, value, title },
                this.getRequestOptions(req),
            )
            .then((response: AxiosResponse<BillsBackend.DocumentResponse>) => {
                const document = documentReducer(response.data.data.document);

                return res.json(document);
            });
    };

    updateDocument: RequestHandler = (req, res, _next) => {
        const { type, value, title } = req.body;
        const { documentId } = req.params;

        return this.request
            .put(
                `${this.getApiPrefix()}/documents/${documentId}`,
                { type, value, title },
                this.getRequestOptions(req),
            )
            .then((response: AxiosResponse<BillsBackend.DocumentResponse>) => {
                const document = documentReducer(response.data.data.document);

                return res.json(document);
            });
    };

    deleteDocument: RequestHandler = (req, res, _next) => {
        const { documentId } = req.params;

        return this.request
            .delete(`${this.getApiPrefix()}/documents/${documentId}`, this.getRequestOptions(req))
            .then((response: AxiosResponse<BillsBackend.DeleteResponse>) => {
                return res.json(response.data.data);
            });
    };

    getBills: RequestHandler = (req, res, next) => {
        return this.request
            .get(`${this.getApiPrefix()}/bills`, this.getRequestOptions(req))
            .then((response: AxiosResponse<BillsBackend.BillsResponse>) => {
                const bills = response.data.data.bills.map(billReducer);

                return res.json({
                    state: response.data.data.state,
                    bills,
                });
            })
            .catch(apiErrorHandler(req, res, next));
    };

    restartSearch: RequestHandler = (req, res, _next) => {
        return this.request
            .post(`${this.getApiPrefix()}/search/bills`, {}, this.getRequestOptions(req))
            .then((_response: AxiosResponse<{}>) => {
                return res.json({});
            });
    };

    createOrder: RequestHandler = (req, res, next) => {
        const { bills } = req.body;

        const payload = { bill_ids: bills };

        return this.request
            .post(`${this.getApiPrefix()}/orders`, payload, this.getRequestOptions(req))
            .then((response: AxiosResponse<BillsBackend.CreateOrderResponse>) => {
                return res.json(response.data.data);
            })
            .catch(apiErrorHandler(req, res, next));
    };

    startOrder: RequestHandler = (req, res, next) => {
        const { orderId } = req.params;
        const body = req.body as IBillPayApiRequest;

        const { payer_full_name, payment_token, return_url, payment_method } = body;
        const mpi_3ds_info = this.fullfillMpiInfo(req, body.mpi_3ds_info);

        const payload = {
            payer_full_name,
            payment_token,
            return_url,
            payment_method,
            mpi_3ds_info,
        };

        return this.request
            .post(
                `${this.getApiPrefix()}/orders/${orderId}/transactions`,
                payload,
                this.getRequestOptions(req),
            )
            .then((response: AxiosResponse<BillsBackend.PayResponse>) => {
                return res.json(response.data.data);
            })
            .catch(apiErrorHandler(req, res, next));
    };

    getTransaction: RequestHandler = (req, res, _next) => {
        const { transactionId } = req.params;

        return this.request
            .get(
                `${this.getApiPrefix()}/transactions/${transactionId}`,
                this.getRequestOptions(req),
            )
            .then((response: AxiosResponse<BillsBackend.PaymentStatusResponse>) => {
                return res.json(response.data.data);
            });
    };

    setNotificationsAgreement: RequestHandler = (req, res, _next) => {
        const { uid } = req.blackbox || {};
        const { agree } = req.body;

        req.log.info(
            { receiveNotifications: { uid, agree } },
            'User selected receive notifications agreement option',
        );

        return res.json({});
    };
}

const bills = new BillsApi();

export const getDocumentsStateMiddleware = bills.getDocumentsStateMiddleware;
export const getBillsStateMiddleware = bills.getBillsStateMiddleware;
export const getDocuments = bills.getDocuments;
export const createDocument = bills.createDocument;
export const updateDocument = bills.updateDocument;
export const deleteDocument = bills.deleteDocument;
export const getBills = bills.getBills;
export const restartSearch = bills.restartSearch;
export const createOrder = bills.createOrder;
export const startOrder = bills.startOrder;
export const getTransaction = bills.getTransaction;
export const setNotificationsAgreement = bills.setNotificationsAgreement;
