import { AxiosRequestConfig, AxiosResponse } from 'axios';
import type { Request } from 'express';

import { BlackboxFamilyInfo, BlackboxUserInfoData, UserInfo } from './types';

import { ApiMiddleware } from '..';
import config from '../../../config';
import logger from '../../../lib/logger';
import { createApiRequest } from '../../../lib/request';

class BlackboxApi extends ApiMiddleware {
    constructor() {
        super();

        this.request = createApiRequest({
            baseURL: config.blackbox.url,
            timeout: config.blackbox.timeout,
            responseType: 'json',
        });
    }

    getRequestOptions = (req: Request): AxiosRequestConfig => {
        const { tvm, blackbox } = req;
        const ticket = tvm?.frontend.tickets?.blackbox?.ticket;

        const headers: AxiosRequestConfig['headers'] = {};

        if (ticket) {
            headers['X-Ya-Service-Ticket'] = ticket;
        }

        if (blackbox?.userTicket) {
            headers['X-Ya-User-Ticket'] = blackbox.userTicket;
        }

        return { headers };
    }

    /*
        Получение списка uid-ов семейной группы
        Ручка: https://nda.ya.ru/t/0pRvnBG24QMqPP
    */
    getFamilyUsers = (req: Request): Promise<string[]> => {
        const familyId = req.blackbox?.raw?.family_info?.family_id;

        return this
            .request('/blackbox', {
                ...this.getRequestOptions(req),
                params: {
                    method: 'family_info',
                    family_id: familyId,
                    format: 'json',
                },
            })
            .then((response: AxiosResponse<BlackboxFamilyInfo>) => {
                return response.data.family[familyId].users.map(({ uid }) => uid);
            });
    }

    /*
        Получение информации о пользователе
        Ручка: https://doc.yandex-team.ru/blackbox/reference/MethodUserInfo.html
    */
    getUserInfo = (req: Request, uids: string[]): Promise<Record<string, UserInfo>> => {
        return this
            .request('/blackbox', {
                ...this.getRequestOptions(req),
                params: {
                    method: 'userinfo',
                    uid: uids.join(','),
                    attributes: 27,
                    regname: 'yes',
                    get_public_name: 'yes',
                    userip: req.headers['x-forwarded-for'] || req.ip,
                    format: 'json',
                },
            })
            .then((response: AxiosResponse<BlackboxUserInfoData>) => {
                const users: Record<string, UserInfo> = {};

                if (!response.data.users) {
                    logger.error({ responseData: response.data }, 'Blackbox userInfo error');

                    throw new Error('Blackbox userInfo error');
                }

                response.data.users.forEach(({ id, attributes, display_name }) => {
                    users[id] = {
                        name: attributes['27'],
                        avatar: display_name.avatar.default,
                        uid: id,
                    };
                });

                return users;
            });
    }
}

export const getFamilyUsers = new BlackboxApi().getFamilyUsers;
export const getUserInfo = new BlackboxApi().getUserInfo;
