import { AxiosRequestConfig, AxiosResponse } from 'axios';
import { Request, RequestHandler, Response } from 'express';

import { ordersReducer } from './reducers/orders';

import { getRoutes } from '../../../../routes';
import { IOrderList, IReceipt, IService } from '../../../../types';
import config from '../../../config';
import { base64Decode } from '../../../helpers';
// import { makeDump } from '../../../lib/dump';
import { saftyParseFloat } from '../../../lib/float';
import logger from '../../../lib/logger';
import { createApiRequest } from '../../../lib/request';
import error from '../../error';
import { apiErrorHandler } from '../../error/apiErrorHandler';
import * as bills from '../bills';
import { getFamilyUsers, getUserInfo } from '../blackbox';
import { ApiMiddleware } from '../index';
import * as trust from '../trust';

const PLUS = 'plus';
const RECEIPTS = 'receipts';
const CURSOR_JOIN = '__';
const LIMIT = 20;

interface IOrdersParams {
    alias: string;
    initiatorUids?: string[];
}

const parseCursor = (cursor?: string) => {
    if (!cursor) {
        return;
    }

    const parts = base64Decode(cursor).split(CURSOR_JOIN);

    return {
        orderIdKeyset: parts[0],
        createdKeyset: parts[1],
    };
};

const getAllServices = (services: Record<string, BunkerService>) => {
    return Object.values(services).reduce<string[]>((aux, { ids }) => {
        aux.push(...ids);

        return aux;
    }, []);
};

class OhioApi extends ApiMiddleware {
    constructor() {
        super();

        this.request = createApiRequest({
            baseURL: config.backend.url,
            timeout: config.backend.timeout,
            responseType: 'json',
        });
    }

    getApiUrlPrefix(req: Request) {
        return `/v1/customer/${req.services.user.current.uid}`;
    }

    getRequestOptions = (req: Request): AxiosRequestConfig => {
        const { tvm, blackbox } = req;
        const ticket = tvm?.frontend.tickets?.backend?.ticket;
        const headers: AxiosRequestConfig['headers'] = {};

        if (ticket) {
            headers['X-Ya-Service-Ticket'] = ticket;
        }

        if (blackbox?.userTicket) {
            headers['X-Ya-User-Ticket'] = blackbox.userTicket;
        }

        return { headers };
    };

    receiptsReducer = ({ url, payments, timestamp }: Backend.Receipt): IReceipt => {
        const amount = payments.map(({ amount }) => amount);
        const type = payments?.[0]?.payment_type;

        return {
            url,
            amount,
            timestamp,
            type,
        };
    };

    getReceiptByPurchaseToken: RequestHandler = (req, res, next) => {
        const purchaseToken = req.body.token;

        return this.request
            .get(`/receipts?purchase_token=${purchaseToken}`, this.getRequestOptions(req))
            .then((response: AxiosResponse<Backend.ReceiptsResponse>) => {
                const receipts = response.data.data.receipts.map(this.receiptsReducer);

                return res.json({ receipts });
            })
            .catch(apiErrorHandler(req, res, next));
    };

    getServices = (req: Request, _res: Response) => {
        const yandexPayEnabled = req.appContext.yandexPayEnabled;

        return this.request
            .get(`${this.getApiUrlPrefix(req)}/services`, {
                ...this.getRequestOptions(req),
                params: {
                    'show-yandexpay': yandexPayEnabled,
                },
            })
            .then((response: AxiosResponse<Backend.ServiceResponse>) => {
                return req.appContext.servicesOrder.reduce<IService[]>((aux, alias) => {
                    const service = req.appContext.services[alias];

                    const exists = response.data.data.services.some((backendService) => {
                        return service.ids.has(backendService.subservice_id);
                    });

                    if (alias === PLUS && req.query.family) {
                        return aux;
                    }

                    if (exists || alias === PLUS) {
                        aux.push({
                            id: service.alias,
                            ...service,
                        });
                    }

                    return aux;
                }, []);
            });
    };

    getBaseData: RequestHandler = (req, res, next) => {
        Promise.all([
            this.getServices(req, res),
            this.fnsBindingMiddleware(req, res),
            bills.getDocumentsStateMiddleware(req, res),
            bills.getBillsStateMiddleware(req, res),
            trust.getPlusBalance(req),
        ])
            .then(([services, fns, documentsState, billsState, plusBalance]) => {
                res.locals.services = services;
                res.locals.fns = fns;
                res.locals.billsState = {
                    documents: documentsState,
                    bills: billsState,
                };

                res.locals.plusBalance = plusBalance;
                next();
            })
            .catch((err) => error(err, req, res, next));
    };

    getOrders = (req: Request, { alias, initiatorUids }: IOrdersParams, after?: string) => {
        const service = alias ? req.appContext.services[alias] : undefined;
        const yandexPayEnabled = req.appContext.yandexPayEnabled;
        const subservice_ids = service
            ? Array.from(service.ids)
            : getAllServices(req.appContext.services);
        const { orderIdKeyset, createdKeyset } = parseCursor(after) || {};

        let params: AxiosRequestConfig['params'] = {};

        if (alias === PLUS) {
            return this.getPlusOrders(req, orderIdKeyset, createdKeyset);
        }

        if (alias === RECEIPTS && req.appContext.fnsEnabled) {
            return this.getFnsOrders(req, orderIdKeyset, createdKeyset);
        }

        if (initiatorUids?.length === 1 && req.blackbox?.uid === initiatorUids[0]) {
            params = { initiator_uid: undefined, 'hide-family-payments': true };
        }

        return this.request
            .get(`${this.getApiUrlPrefix(req)}/orders`, {
                ...this.getRequestOptions(req),
                params: {
                    subservice_ids,
                    limit: LIMIT,
                    order_id_keyset: orderIdKeyset,
                    created_keyset: createdKeyset,
                    initiator_uid: initiatorUids,
                    'show-yandexpay': yandexPayEnabled,
                    'show-holds': true,
                    ...params,
                },
            })
            .then((response: AxiosResponse<Backend.OrdersResponse>) => {
                // makeDump(
                //     'src/server/middleware/api/ohio/reducers/__tests__/data/list',
                //     response.data.data,
                //     req.appContext,
                //     () => ordersReducer(response.data.data, req.appContext),
                // );

                return ordersReducer(response.data.data, req.appContext);
            });
    };

    getPlusOrders = (
        req: Request,
        orderIdKeyset: string | undefined,
        createdKeyset: string | undefined,
    ) => {
        return this.request
            .get(`${this.getApiUrlPrefix(req)}/yandex_account_orders`, {
                ...this.getRequestOptions(req),
                params: {
                    limit: LIMIT,
                    order_id_keyset: orderIdKeyset,
                    created_keyset: createdKeyset,
                },
            })
            .then((response: AxiosResponse<Backend.OrdersResponse>) => {
                // makeDump(
                //     'src/server/middleware/api/ohio/reducers/__tests__/data/plus',
                //     response.data.data,
                //     req.appContext,
                //     () => ordersReducer(response.data.data, req.appContext),
                // );

                return ordersReducer(response.data.data, req.appContext, { notIgnoreTrust: true });
            });
    };

    familyMiddleware: RequestHandler = async (req, res, next) => {
        const { uid, raw } = req.blackbox || {};
        const { admin_uid: adminUid } = raw?.family_info || {};

        // Проверяем, что текущий пользователь является админом семьи
        res.locals.isFamilyAdmin = adminUid === uid;

        if (res.locals.isFamilyAdmin) {
            const actualFamilyUids = await getFamilyUsers(req);

            res.locals.familyUsers = await this.getFamilyPayUsers(req)
                .then((uids) => {
                    // Находим пересечение актуального состава семьи и uid-ов которые платили семейной картой
                    // Чтобы в листинге не было платежей бывших участников семьи
                    const actualFamilyUidsSet = new Set(actualFamilyUids);

                    res.locals.familyUsersUids = uids.filter((item) =>
                        actualFamilyUidsSet.has(item),
                    );

                    if (uids && uids.length) {
                        return getUserInfo(req, uids);
                    }
                })
                .catch((error) => {
                    if (error?.config?.headers) {
                        delete error.config.headers;
                    }
                    logger.error({ error }, `Error while getting family info: ${error.message}`);
                });
        }

        next();
    };

    getOrdersMiddleware: RequestHandler = async (req, res, next) => {
        const familyMode = req.query.family as string;
        const familyFilter = req.query.familyFilter as string;
        const { isFamilyAdmin, familyUsersUids } = res.locals;
        let alias = req.query.alias as string;
        let initiatorUids = familyMode && isFamilyAdmin ? familyUsersUids : [];

        if (req.path === getRoutes(config.baseUrl).receipts) {
            alias = 'receipts';
        }

        logger.debug({ attributes: req.blackbox?.raw.attributes }, 'Attributes');

        /*
            Если семейной картой еще не платили, то админу не показываем платежи.
            Для этого – передаем его uid в initiator_uid
        */
        if (familyMode && isFamilyAdmin && familyUsersUids?.length === 0) {
            initiatorUids = [req.blackbox?.uid];
        } else if (isFamilyAdmin && familyFilter && familyUsersUids.includes(familyFilter)) {
            initiatorUids = [familyFilter];
        }

        return this.getOrders(req, { alias, initiatorUids })
            .then((orders) => {
                res.locals.orders = orders;

                if (req.query.alias === 'receipts') {
                    return res.redirect(getRoutes(config.baseUrl).receipts);
                }

                next();
            })
            .catch((err) => error(err, req, res, next));
    };

    getOrdersApi: RequestHandler = (req, res, next) => {
        const { alias, after } = req.query;

        const initiatorUids = req.query.initiatorUids
            ? (req.query.initiatorUids as string).split(',')
            : undefined;

        return this.getOrders(req, { alias: alias as string, initiatorUids }, after as string)
            .then((orders) => {
                return res.json(orders);
            })
            .catch(apiErrorHandler(req, res, next));
    };

    /*
        Получение списка членов семьи, которые пользовались семейной оплатой по карте текущего пользователя
        Docs: https://nda.ya.ru/t/027GtQnQ4QMpPi
    */
    getFamilyPayUsers = (req: Request): Promise<string[]> => {
        return this.request(
            `${this.getApiUrlPrefix(req)}/familypay_users`,
            this.getRequestOptions(req),
        ).then((response: AxiosResponse<Backend.FamilyResponse>) => {
            return response.data.data.familypay_users.map(({ initiator_uid }) =>
                String(initiator_uid),
            );
        });
    };

    getPaymentById: RequestHandler = (req, res, next) => {
        const { paymentId } = req.params;
        const { amount, currency, cardMask } = req.query;

        const fallbackP = {
            id: paymentId,
            refunds: [],
            items: [],
            orderId: paymentId,
            status: 'paid',
            total: amount,
            plus: 0,
            currency,
            payments: [
                {
                    method: 'card',
                    price: amount,
                    account: cardMask,
                    currency,
                },
            ],
            refundOnly: false,
            checksEnabled: false,
            service: {
                id: 'unknown',
                alias: 'unknown',
                name: '',
                i18nKeyName: '',
                iconUrl: '',
            },
        };

        return this.request
            .get(`${this.getApiUrlPrefix(req)}/payment`, {
                ...this.getRequestOptions(req),
                params: {
                    trust_payment_id: paymentId,
                },
            })
            .then((response) => {
                return ordersReducer(response.data.data, req.appContext, { notIgnoreTrust: true });
            })
            .then(({ edges }) => {
                res.locals.order = edges[0] || fallbackP;

                next();
            })
            .catch((err) => error(err, req, res, next));
    };

    // Получение статуса подключения электрочеков
    fnsBindingMiddleware = async (req: Request, _res: Response) => {
        if (!req.appContext.fnsEnabled) {
            return;
        }

        const fnsData = await this.request
            .get(`${this.getApiUrlPrefix(req)}/fns_binding_status`, {
                ...this.getRequestOptions(req),
            })
            .then((response: AxiosResponse<Backend.FnsBinding>) => {
                const { status, urls } = response.data;

                return { status, urls };
            })
            .catch((error) => {
                if (error?.config?.headers) {
                    delete error.config.headers;
                }

                logger.error({ error }, 'FNS binding error');
            });

        let amount;

        if (fnsData && fnsData.status === 'has_binding') {
            const rawAmount = await this.getCashbackBalance(req).catch((error) => {
                logger.error({ error }, 'getCashbackBalance error');

                return '0';
            });

            amount = Math.floor(saftyParseFloat(rawAmount));
        }

        return { ...fnsData, amount };
    };

    getFnsOrders = (
        req: Request,
        orderIdKeyset: string | undefined,
        createdKeyset: string | undefined,
    ): Promise<IOrderList> => {
        return this.request
            .get(`${this.getApiUrlPrefix(req)}/fns_orders`, {
                ...this.getRequestOptions(req),
                params: {
                    limit: LIMIT,
                    order_id_keyset: orderIdKeyset,
                    created_keyset: createdKeyset,
                },
            })
            .then((response: AxiosResponse<Backend.OrdersResponse>) => {
                logger.debug({ data: response.data.data }, 'FNS response');

                return ordersReducer(response.data.data, req.appContext, { isReceipt: true });
            });
    };

    getCashbackBalance = (req: Request): Promise<string> => {
        return this.request
            .get(`${this.getApiUrlPrefix(req)}/cashback_balance`, this.getRequestOptions(req))
            .then((response: AxiosResponse<Backend.CashbackBalanceResponse>) => {
                return response.data.data.amount;
            });
    };
}

export const getReceiptByPurchaseToken = new OhioApi().getReceiptByPurchaseToken;
export const getServices = new OhioApi().getServices;
export const getOrdersMiddleware = new OhioApi().getOrdersMiddleware;
export const getOrdersApi = new OhioApi().getOrdersApi;
export const getPaymentById = new OhioApi().getPaymentById;
export const familyMiddleware = new OhioApi().familyMiddleware;
export const fnsBindingMiddleware = new OhioApi().fnsBindingMiddleware;
export const getBaseData = new OhioApi().getBaseData;
